import axios from 'axios';
import cookie from 'js-cookie';

import { isUrl } from '@lib/string';

import { IFileUpload } from '../interfaces';

interface IResponse<T> {
  status: number;
  data: T;
}

export const TOKEN = 'token';

export abstract class APIRequest {
  static API_ENDPOINT: string = '';

  getBaseApiEndpoint() {
    return process.env.API_ENDPOINT || process.env.NEXT_PUBLIC_API_ENDPOINT;
  }

  request(
    url: string,
    method?: string,
    body?: any,
    headers?: { [key: string]: string },
    options = {}
  ): Promise<IResponse<any>> {
    const verb = (method || 'get').toUpperCase();

    const updatedHeader = {
      'Content-Type': 'application/json',
      // TODO - check me
      Authorization: cookie.get(TOKEN) || '',
      ...(headers || {})
    };
    const baseApiEndpoint = this.getBaseApiEndpoint();

    return axios({
      method: verb,
      url: isUrl(url) ? url : `${baseApiEndpoint}${url}`,
      data: body ? JSON.stringify(body) : undefined,
      headers: updatedHeader,
      // cache: 'force-cache',
      ...options
    })
      .then((resp) => resp.data)
      .catch((e) => {
        const { response } = e;
        if (response.status === 401) {
          if (typeof window !== 'undefined') {
            window.location.href = '/';
          }

          throw new Error('Forbidden in the action!');
        }

        throw response.data;
      });
  }

  buildUrl(baseUrl: string, params?: { [key: string]: any }) {
    if (!params) {
      return baseUrl;
    }

    const queryString = Object.keys(params)
      .map((k) => {
        if (Array.isArray(params[k])) {
          return params[k]
            .map(
              (param: any) => `${encodeURIComponent(k)}=${encodeURIComponent(param)}`
            )
            .join('&');
        }
        return `${encodeURIComponent(k)}=${encodeURIComponent(params[k])}`;
      })
      .join('&');
    return `${baseUrl}?${queryString}`;
  }

  get(url: string, headers?: { [key: string]: any }, options = {}) {
    return this.request(url, 'get', null, headers, options);
  }

  post(
    url: string,
    data?: any,
    headers?: { [key: string]: string },
    options = {}
  ) {
    return this.request(url, 'post', data, headers, options);
  }

  put(url: string, data?: any, headers?: { [key: string]: string }) {
    return this.request(url, 'put', data, headers);
  }

  del(url: string, data?: any, headers?: { [key: string]: string }) {
    return this.request(url, 'delete', data, headers);
  }

  async upload({
    url,
    body,
    file,
    type,
    fieldName = 'file',
    onProgress = () => {}
  }: IFileUpload) {
    const { data } = await this.post(url, {
      ...(body || {}),
      type
    });
    const {
      uploadUrl, fields = [], method = 'POST', ...rests
    } = data;
    return new Promise((resolve, reject) => {
      const req = new XMLHttpRequest();

      req.upload.addEventListener('progress', (event) => {
        if (event.lengthComputable) {
          onProgress({
            percentage: (event.loaded / event.total) * 100
          });
        }
      });

      req.addEventListener('load', () => {
        const success = req.status >= 200 && req.status < 300;
        const { response } = req;
        if (!success) {
          return reject(response);
        }
        return resolve({
          ...rests,
          uploadUrl,
          status: success
        });
      });

      req.upload.addEventListener('error', () => {
        reject(req.response);
      });

      const formData = new FormData();
      if (fields && Object.keys(fields).length) {
        Object.keys(fields).forEach((key) => {
          formData.append(key, fields[key]);
        });
      }
      // append file at last is required for AWS upload
      formData.append(fieldName, file, file.name);
      req.responseType = 'json';
      req.open(method || 'POST', uploadUrl);
      req.send(formData);
    });
  }
}
