'use client';

import {
  createContext,
  useContext,
  useState,
  ReactNode,
  useEffect,
  useMemo,
  useCallback
} from 'react';

import { signOut, useSession } from 'next-auth/react';
import { IProfile, Res } from '../interfaces';
import { authService, profileService } from '../services';

export interface IProfileContext {
  current: IProfile;
  fetching: boolean;
  loadProfile: () => void;
  loggedIn: boolean;
}

const ProfileContext = createContext<IProfileContext>({
  current: null as any,
  fetching: true,
  loadProfile: () => {},
  loggedIn: false
});

export function ProfileProvider({ children }: { children: ReactNode }) {
  const session = useSession();
  const [fetching, setFetching] = useState(true);
  const [current, setCurrent] = useState<IProfile>(null as any);

  const loadProfile = useCallback(async () => {
    if (session && session.data?.accessToken) {
      try {
        setFetching(true);
        const { data } = await profileService.getMe(session.data.accessToken);
        // reset auth token. should replace with session
        authService.setToken(session.data.accessToken);
        setCurrent(data);
      } catch (e) {
        const error = await e as Res;
        if (error && error.statusCode === 403) {
          authService.clearToken();
          await signOut();
          window.location.href = '/';
        }
      } finally {
        setFetching(false);
      }
    } else {
      setFetching(false);
    }
  }, [session]);

  useEffect(() => {
    loadProfile();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [session]);

  const profileValue = useMemo(
    () => ({
      current,
      fetching,
      loadProfile,
      loggedIn: session.status === 'authenticated'
    }),
    [current, fetching, session, loadProfile]
  );

  return (
    <ProfileContext.Provider value={profileValue}>
      {children}
    </ProfileContext.Provider>
  );
}

export const useProfile = () => useContext(ProfileContext);
