'use client';

import { defaultSettings } from '@lib/utils';
import { useSearchParams } from 'next/navigation';
import {
  createContext,
  useContext,
  ReactNode,
  useMemo,
  useReducer,
  useCallback,
  useState,
  useEffect
} from 'react';
import { toast } from 'react-toastify';
import { PublicSettings } from 'src/interfaces';

export interface IMainLayoutContext {
  login: () => void;
  register: () => void;
  closePopup: () => void;
  forgotPassword: () => void;
  toggleUseSettingDrawer: () => void;
  openLoginPopup: boolean;
  openRegisterPopup: boolean;
  openForgotPasswordPopup: boolean;
  openUserSettingDrawer: boolean;
  publicSettings: Partial<typeof defaultSettings>;
}

const MainLayoutContext = createContext<IMainLayoutContext>({
  login: () => { },
  register: () => { },
  closePopup: () => { },
  forgotPassword: () => { },
  toggleUseSettingDrawer: () => { },
  openLoginPopup: false,
  openRegisterPopup: false,
  openForgotPasswordPopup: false,
  openUserSettingDrawer: false,
  publicSettings: defaultSettings
});

interface Action {
  type: string;
  payload?: any;
}

const initializer = {
  openLoginPopup: false,
  openRegisterPopup: false,
  openForgotPasswordPopup: false,
  openUserSettingDrawer: false
};

const reducers = (state: typeof initializer, action: Action) => {
  switch (action.type) {
    case 'login': {
      return {
        ...state,
        openLoginPopup: true,
        openRegisterPopup: false,
        openForgotPasswordPopup: false
      };
    }
    case 'register': {
      return {
        ...state,
        openLoginPopup: false,
        openRegisterPopup: true,
        openForgotPasswordPopup: false
      };
    }
    case 'forgotPassword': {
      return {
        ...state,
        openLoginPopup: false,
        openRegisterPopup: false,
        openForgotPasswordPopup: true
      };
    }
    case 'close': {
      return {
        ...state,
        openLoginPopup: false,
        openRegisterPopup: false,
        openForgotPasswordPopup: false
      };
    }
    case 'toggleUseSettingDrawer': {
      return {
        ...state,
        openUserSettingDrawer: !state.openUserSettingDrawer
      };
    }
    default: {
      return {
        ...state
      };
    }
  }
};

export function MainLayoutProvider({ children, settings }: { children: ReactNode, settings: Record<string, any> }) {
  const [state, dispatch] = useReducer(reducers, initializer);
  const [publicSettings] = useState<Partial<PublicSettings>>(settings);
  const params = useSearchParams();

  const closePopup = useCallback(() => dispatch({ type: 'close' }), [dispatch]);
  const login = useCallback(() => dispatch({ type: 'login' }), [dispatch]);
  const register = useCallback(
    () => dispatch({ type: 'register' }),
    [dispatch]
  );
  const forgotPassword = useCallback(
    () => dispatch({ type: 'forgotPassword' }),
    [dispatch]
  );
  const toggleUseSettingDrawer = useCallback(
    () => dispatch({ type: 'toggleUseSettingDrawer' }),
    [dispatch]
  );

  useEffect(() => {
    const message = params.get('message');
    if (message) {
      toast.success(decodeURIComponent(message));
    }
  }, [params]);

  const themeValue = useMemo(
    () => ({
      closePopup,
      login,
      register,
      forgotPassword,
      toggleUseSettingDrawer,
      publicSettings,
      ...state
    }),
    [closePopup, login, register, forgotPassword, toggleUseSettingDrawer, publicSettings, state]
  );

  return (
    <MainLayoutContext.Provider value={themeValue}>
      {children}
    </MainLayoutContext.Provider>
  );
}

export const useMainThemeLayout = () => useContext(MainLayoutContext);
