import { DefaultSession, NextAuthOptions } from 'next-auth';
import CredentialsProvider from 'next-auth/providers/credentials';
import { IProfile } from 'src/interfaces';

declare module 'next-auth' {
  interface User extends IProfile {
    token: string;
  }

  interface Session extends DefaultSession {
    accessToken: string;
    user: any;
  }
}

export const authOptions: NextAuthOptions = {
  providers: [
    CredentialsProvider({
      // The name to display on the sign in form (e.g. "Sign in with...")
      name: 'Credentials',
      type: 'credentials',
      // `credentials` is used to generate a form on the sign in page.
      // You can specify which fields should be submitted, by adding keys to the `credentials` object.
      // e.g. domain, username, password, 2FA token, etc.
      // You can pass any HTML attribute to the <input> tag through the object.
      credentials: {
        username: { label: 'Username', type: 'text', placeholder: 'Username' },
        password: {
          label: 'Password',
          type: 'password',
          placeholder: 'Password'
        }
      },
      async authorize(credentials) {
        try {
          const resp = await fetch(`${process.env.API_ENDPOINT}/auth/login`, {
            method: 'POST',
            body: JSON.stringify(credentials),
            headers: { 'Content-Type': 'application/json' }
          }).then((res) => res.json());

          return {
            ...resp.data.user,
            token: resp.data.token
          };
        } catch (e) {
          return null;
        }
      }
    })
  ],
  pages: {
    signIn: '/'
  },
  session: { strategy: 'jwt', maxAge: 60 * 60 * 24 * 7 },
  callbacks: {
    signIn: ({ user }) => {
      if (user.token) return true;

      return false;
    },
    jwt({ token, user }) {
      if (user && user.token) {
        // eslint-disable-next-line no-param-reassign
        token.accessToken = user.token;
        // eslint-disable-next-line no-param-reassign
        token.user = {
          email: user.email,
          username: user.username,
          avatar: user.avatar,
          _id: user._id
        };
      }
      return token;
    },
    async session({ session, token }) {
      // eslint-disable-next-line no-param-reassign
      session.accessToken = token.accessToken as string;
      // eslint-disable-next-line no-param-reassign
      session.user = token.user;
      return session;
    }
  }
};
