/* eslint-disable no-restricted-globals */

'use client';

import { useEffect, useMemo, useRef } from 'react';
import { IVideo } from 'src/interfaces';
import { videoService } from 'src/services';
import classNames from 'classnames';
import fluidPlayer from 'fluid-player';
import { isIOS, isMobileOnly } from 'react-device-detect';
import { getVideoScreenOrientation } from '@lib/video';
import { angle } from '@lib/utils';
import s from './video-player.module.scss';
import { useVideoPlayer } from './video-player-provider';

export default function VideoPlayerScreen() {
  const {
    video, settings
  } = useVideoPlayer();
  const vjsPlayer = useRef<HTMLDivElement | null>(null);
  const videoRef = useRef<IVideo>();
  // eslint-disable-next-line no-undef
  const playerRef = useRef<FluidPlayerInstance>();

  const isFullscreen = useRef(false);

  const isEmbed = useMemo(() => !!video!.embeddedCode, [video]);
  const viewable = useMemo(() => false, [video]);
  const canPlay = useMemo(
    () => video!.video && video.video.url && viewable,
    [video, viewable]
  );

  const addView = async (videoId: string) => {
    await videoService.addView(videoId);
  };

  const onPlay = async () => {
    if (videoRef.current && videoRef.current._id) {
      if (!videoRef.current.played) await videoService.addView(videoRef.current._id);
      videoRef.current.played = true;
    }
    if (window.innerWidth <= 768) {
      try {
        await (screen.orientation as any).lock('landscape');
      } catch (err) {
        // eslint-disable-next-line no-console
        console.warn('Device does not support landscape rotation.');
      }
    }
  };

  useEffect(() => {
    const initPlayer = async () => {
      if (!videoRef.current || video!._id !== videoRef.current!._id) {
        // player && player.removeEventListener('play', onPlay);

        if (isEmbed) {
          if (viewable) {
            vjsPlayer.current!.innerHTML = /^<iframe/g.test(video!.embeddedCode)
              ? video!.embeddedCode
              : `<iframe allowfullscreen webkitallowfullscreen mozallowfullscreen oallowfullscreen msallowfullscreen src="${(
                video.embeddedCode || ''
              ).replace(
                'http://',
                'https://'
              )}" frameborder="0" width="100%" height="540" scrolling="no"></iframe>`;
            addView(video._id);
          } else {
            const player = document.createElement('video');
            player.playsInline = true;
            player.className = 'w-full h-full absolute inset-y-0 start-0';
            vjsPlayer.current && vjsPlayer.current.append(player);
            // eslint-disable-next-line no-mixed-operators
            const thumbnail = (video.thumbnails && video.thumbnails[0]) || '/images/poster.png';
            player.poster = thumbnail;
          }
        } else {
          const player = document.createElement('video');
          player.playsInline = true;
          player.className = 'w-full h-full absolute inset-y-0 start-0';
          vjsPlayer.current && vjsPlayer.current.append(player);

          // eslint-disable-next-line no-undef

          const source = document.createElement('source');
          source.src = video!.video.url;
          source.type = 'video/mp4';
          player.append(source);
          player.controls = true;
          player.muted = true;
          const posterImage = video.video.poster
            ? video.video.poster.url
            : '/images/poster.png';
          player.poster = posterImage;

          playerRef.current = fluidPlayer(player, {
            layoutControls: {
              primaryColor: '#f97316',
              fillToContainer: true,
              allowTheatre: false,
              posterImage,
              miniPlayer: {
                enabled: false
              },
              // logo: {
              //   imageUrl: '/images/logo.png',
              //   position: 'top right',
              //   clickUrl: null,
              //   opacity: 1,
              //   imageMargin: '0 10px'
              // },
              playButtonShowing: true
            }
          });

          player.addEventListener('play', onPlay);
          player.addEventListener('timeupdate', () => {
            if (player.paused) {
              const playButton = document.querySelector('.fluid_initial_play');
              if (playButton) {
                playButton.classList.add('custom-play');
                playButton.addEventListener(
                  'click',
                  () => {
                    player.play();
                  },
                  { once: true }
                );
              }
            }
          });

          player.addEventListener('play', () => {
            const playButton = document.querySelector('.fluid_initial_play');
            if (playButton) {
              playButton.classList.remove('custom-play');
            }
          });

          const orientation = getVideoScreenOrientation(video.video);
          const fullscreenButton = document.querySelector('.fluid_control_fullscreen');
          const wrapper = document.querySelector('.fluid_video_wrapper');

          if (fullscreenButton) {
            const customFullscreenButton = fullscreenButton.cloneNode(true);
            customFullscreenButton.addEventListener('click', () => {
              if (isFullscreen.current) {
                if (isMobileOnly && orientation === 'landscape') {
                  wrapper && wrapper.classList.remove('pseudo_fullscreen', orientation);
                  if (orientation === 'landscape') {
                    document.documentElement.classList.remove('overflow-hidden');
                  }
                  isFullscreen.current = false;
                } else {
                  playerRef.current?.toggleFullScreen(false);
                  isFullscreen.current = false;
                }
                return;
              }

              const currentAngle = angle();

              if (isMobileOnly && orientation === 'landscape' && currentAngle !== 90 && currentAngle !== 270 && currentAngle !== -90) {
                if (currentAngle !== 90 && currentAngle !== 270 && currentAngle !== -90) {
                  wrapper && wrapper.classList.add('pseudo_fullscreen', orientation);
                  if (orientation === 'landscape') {
                    document.documentElement.classList.add('overflow-hidden');
                  }
                  isFullscreen.current = true;
                } else {
                  wrapper && wrapper.classList.add('pseudo_fullscreen');
                  isFullscreen.current = true;
                }
                // Fix progress bar
                const onProgressbarMouseMove = (event: any) => {
                  // get event offset Y
                  let clickedY = event.clientY;

                  if (typeof event.touches !== 'undefined' && typeof event.touches[0] !== 'undefined') {
                    clickedY = event.touches[0].clientY;
                  }

                  if (clickedY) {
                    setTimeout(() => {
                      const currentTime = (clickedY / window.innerHeight) * player.duration;
                      player.currentTime = currentTime;
                      const currentProgressTag = document.querySelector('.xprogress .fluid_controls_currentprogress') as HTMLDivElement;
                      if (currentProgressTag) {
                        currentProgressTag.style.width = `${(currentTime / player.duration) * 100}%`;
                      }
                    }, 50);
                  }
                };

                const onProgressbarMouseUp = (event: any) => {
                  // get event offset Y
                  const clickedY = event.clientY;
                  if (clickedY) {
                    setTimeout(() => {
                      const currentTime = (clickedY / window.innerHeight) * player.duration;
                      const isPLaying = !(player.currentTime > 0 && !player.paused && !player.ended && player.readyState > 2);
                      if (isPLaying) {
                        player.pause();
                      }
                      player.currentTime = currentTime;
                      if (isPLaying) {
                        player.play();
                      }
                    }, 50);
                    // const currentProgressTag = document.querySelector('fluid_controls_currentprogress') as HTMLDivElement;
                    // if (currentProgressTag) {
                    //   currentProgressTag.style.width = `${(currentTime / player.duration) * 100}%`;
                    // }
                  }
                };
                const progress = document.querySelector('.fluid_controls_progress_container');
                if (progress) {
                  const customProgres = progress.cloneNode(true) as HTMLDivElement;
                  customProgres.classList.add('xprogress');
                  const controls = document.querySelector('.fluid_controls_container');
                  if (controls) {
                    controls.appendChild(customProgres);
                  }
                  customProgres.removeEventListener('mousemove', onProgressbarMouseMove);
                  customProgres.removeEventListener('touchmove', onProgressbarMouseMove);
                  customProgres.removeEventListener('mouseup', onProgressbarMouseUp);
                  customProgres.removeEventListener('touchend', onProgressbarMouseUp);

                  customProgres.addEventListener('mouseup', onProgressbarMouseUp);
                  customProgres.addEventListener('touchend', onProgressbarMouseUp, { passive: true });
                  customProgres.addEventListener('mousemove', onProgressbarMouseMove);
                  customProgres.addEventListener('touchmove', onProgressbarMouseMove, { passive: true });
                }
              } else {
                playerRef.current?.toggleFullScreen(true);
                isFullscreen.current = true;
              }
            });
            const rightContainer = document.querySelector('.fluid_controls_right');
            if (rightContainer) {
              rightContainer.replaceChild(customFullscreenButton, fullscreenButton);
            }

            if (isIOS) {
              window.addEventListener('orientationchange', () => {
                if (isFullscreen.current) {
                  const currentAngle = angle();
                  if (isFullscreen.current && (currentAngle === 90 || currentAngle === 270 || currentAngle === -90)) {
                    wrapper && wrapper.classList.remove('pseudo_fullscreen', 'landscape');
                    if (orientation === 'landscape') {
                      document.documentElement.classList.remove('overflow-hidden');
                    }
                    playerRef.current?.toggleFullScreen(true);
                  } else {
                    // wrapper && wrapper.classList.add('landscape');
                    isFullscreen.current = false;
                    playerRef.current?.toggleFullScreen(false);
                  }
                }
              });
            } else if (screen && screen.orientation) {
              // addEventListener('orientationchange') is not a user interaction on Android
              screen.orientation.onchange = () => {
                if (isFullscreen.current) {
                  const currentAngle = angle();
                  if (isFullscreen.current && (currentAngle === 90 || currentAngle === 270 || currentAngle === -90)) {
                    wrapper && wrapper.classList.remove('pseudo_fullscreen', 'landscape');
                    if (orientation === 'landscape') {
                      document.documentElement.classList.remove('overflow-hidden');
                    }
                    playerRef.current?.toggleFullScreen(true);
                  } else {
                    // wrapper && wrapper.classList.add('landscape');
                    isFullscreen.current = false;
                    playerRef.current?.toggleFullScreen(false);
                  }
                }
              };
            }
          }
        }
      }

      videoRef.current = video;
    };

    initPlayer();
  }, [video, settings, isEmbed, canPlay, viewable]);

  return (

    <div
      ref={vjsPlayer}
      className={classNames(
        s.embedFrame,
        'bg-black/10 inset-0 w-full h-full absolute'
      )}
    />
  );
}
