'use client';

import { defaultSettings } from '@lib/utils';
import {
  ReactNode,
  createContext,
  useContext,
  useMemo,
  useReducer,
  useState
} from 'react';
import { IVideo, PublicSettings } from 'src/interfaces';

interface Props {
  video: IVideo;
  embedPlayer: boolean;
  children: ReactNode;
  settings?: Partial<PublicSettings>;
}

interface Values {
  video: IVideo;
  embedPlayer: boolean;
  settings: Partial<PublicSettings>;
  setVideo: Function;
  toggleComment:() => void;
}

const VideoPlayerContext = createContext<Values>({
  video: {} as IVideo,
  embedPlayer: false,
  settings: defaultSettings,
  setVideo: () => {},
  toggleComment: () => {}
});

interface Action {
  type: string;
  payload?: any;
}

const initializer = {
  visibleComment: false
};

const reducers = (state: typeof initializer, action: Action) => {
  switch (action.type) {
    case 'SHOW_COMMENT': {
      return {
        ...state,
        visibleComment: !state.visibleComment
      };
    }
    default: {
      return {
        ...state
      };
    }
  }
};

export default function VideoPlayerProvider({
  video: originVideo,
  children,
  embedPlayer,
  settings = defaultSettings,
  ...props
}: Props) {
  const [state, dispatch] = useReducer(reducers, initializer);
  const [video, setVideo] = useState(originVideo);
  const value = useMemo(
    () => ({
      video,
      embedPlayer,
      settings,
      ...props,
      setVideo,
      toggleComment: () => dispatch({ type: 'SHOW_COMMENT' })
    }),
    [video, embedPlayer, settings, props, setVideo]
  );

  return (
    <VideoPlayerContext.Provider value={value}>
      {children}
    </VideoPlayerContext.Provider>
  );
}

export const useVideoPlayer = () => useContext(VideoPlayerContext);
