'use client';

import {
  Formik, Form, Field, FieldProps
} from 'formik';
import { FormItem, Input, VideoFileUpload } from '@components/ui/form';
import * as yup from 'yup';
import { videoService } from '@services/video.service';
import { toast } from 'react-toastify';
import { getResponseError } from '@lib/utils';
import { useRouter } from 'next/navigation';
import { IVideo, VideoUpload as IVideoUpload } from 'src/interfaces';
import { useState } from 'react';
import { CategoriesSelect, TagSelect } from '@components/ui/form/select';

const schema = yup.object().shape({
  title: yup.string().required('Required'),
  description: yup.string().optional()
});

interface Props {
  video?: IVideo | undefined;
}

export default function VideoUpload({ video = undefined }: Props) {
  const router = useRouter();
  const [progress, setProgress] = useState(0);

  const onProgress = ({ percentage }: { percentage: number }) => {
    setProgress(percentage);
  };

  return (
    <Formik
      initialValues={{
        title: video?.title || '',
        file: null,
        description: video?.description || '',
        tags: video?.tags || [],
        categoryIds: video?.categoryIds || []
      }}
      validationSchema={schema}
      validateOnChange
      validateOnBlur
      validateOnMount
      enableReinitialize
      onSubmit={async (values, actions) => {
        try {
          if (!video && !values.file) {
            toast.error('Missing File');
            return;
          }

          const formData = {
            title: values.title,
            description: values.description,
            tags: values.tags,
            categoryIds: values.categoryIds
          } as IVideoUpload;
          if (values.file) {
            const res = (await videoService.uploadMainVideo(
              values.file,
              onProgress
            )) as any;
            formData.fileId = res.fileId;
          }
          video
            ? await videoService.update(video._id, formData)
            : await videoService.create(formData);
          toast.success(video ? 'Save changed!' : 'Upload Successfully. Please wait for approval by the Administrator.');
          router.push('/account/my-upload');
          router.refresh();
        } catch (e) {
          const error = await e;
          toast.error(getResponseError(error));
        } finally {
          actions.setSubmitting(false);
        }
      }}
    >
      {({ handleSubmit, isSubmitting, values }) => (
        <Form
          onSubmit={handleSubmit}
          onKeyUp={(event) => {
            if (event.keyCode === 13) {
              event.preventDefault();
            }
          }}
        >
          <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-x-5 pb-10 mb-5 border-b dark:border-slate-700">
            <FormItem classnames="col-span-2 col-start-1 sm:col-span-1">
              <Field name="title">
                {({ field, meta }: FieldProps) => (
                  <div>
                    <FormItem.Label required>Title</FormItem.Label>
                    <Input placeholder="Title" {...field} />
                    {meta.touched && !!meta.error && (
                    <FormItem.Error>{meta.error}</FormItem.Error>
                    )}
                  </div>
                )}
              </Field>
            </FormItem>
            <FormItem classnames="col-span-2">
              <Field name="description">
                {({ field, meta }: FieldProps) => (
                  <div>
                    <FormItem.Label>Description</FormItem.Label>
                    <Input.TextArea
                      type="textarea"
                      placeholder="Description"
                      {...field}
                      rows={4}
                    />
                    {meta.touched && !!meta.error && (
                    <FormItem.Error>{meta.error}</FormItem.Error>
                    )}
                  </div>
                )}
              </Field>
            </FormItem>
            <FormItem classnames="col-span-2">
              <Field name="tags">
                {({ field, meta, form }: FieldProps) => (
                  <div>
                    <FormItem.Label>Tags</FormItem.Label>
                    <TagSelect
                      {...field}
                      onChange={(value: string[]) => form.setFieldValue('tags', value)}
                    />
                    {meta.touched && !!meta.error && (
                    <FormItem.Error>{meta.error}</FormItem.Error>
                    )}
                  </div>
                )}
              </Field>
            </FormItem>
            <FormItem classnames="col-span-2">
              <Field name="categoryIds">
                {({ field, meta, form }: FieldProps) => (
                  <div>
                    <FormItem.Label>Categories</FormItem.Label>
                    <CategoriesSelect
                      {...field}
                      options={[]}
                      onChange={(value: string[]) => form.setFieldValue('categoryIds', value)}
                    />
                    {meta.touched && !!meta.error && (
                    <FormItem.Error>{meta.error}</FormItem.Error>
                    )}
                  </div>
                )}
              </Field>
            </FormItem>
            {video?.video?.url && !values.file && (
            <FormItem classnames="w-full sm:w-1/2 col-span-2">
              <video src={video.video.url} controls playsInline />
            </FormItem>
            )}
            <FormItem classnames="w-full text-gray-500 col-span-2">
              <Field name="file">
                {({ meta, form }: FieldProps) => (
                  <div>
                    <FormItem.Label>Upload</FormItem.Label>
                    <VideoFileUpload
                      fileList={[]}
                      progress={progress}
                      onChange={(value) => form.setFieldValue('file', value)}
                    />
                    {meta.touched && !!meta.error && (
                    <FormItem.Error>{meta.error}</FormItem.Error>
                    )}
                  </div>
                )}
              </Field>
            </FormItem>
          </div>
          <div className="text-center">
            <button
              type="submit"
              disabled={isSubmitting}
              className="rounded-md bg-primary px-8 py-2.5 text-sm font-semibold leading-6 text-white shadow-sm focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-orange-600"
            >
              Upload
            </button>
          </div>
        </Form>
      )}
    </Formik>
  );
}
