'use client';

import { CloudArrowUpIcon } from '@heroicons/react/24/solid';
import { generateUuid } from '@lib/string';
import { ChangeEvent, useState } from 'react';
import { toast } from 'react-toastify';
import { FileList } from 'src/interfaces';

interface Props {
  fileList: FileList;
  onChange: (file: File, fileList: FileList) => void;
  progress: number;
  multiple?: boolean;
}

export function VideoFileUpload({
  fileList = [], multiple = false, onChange, progress = 0
}: Props) {
  const [files, setFiles] = useState<FileList>(fileList);

  const handleChange = (e: ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files && e.target.files[0];
    if (!file) return;

    const isMaxSize = file.size / 1024 / 1024 > (Number(process.env.MAX_SIZE_UPLOAD_VIDEO || 2000));
    if (isMaxSize) {
      toast.error(`Video must be smaller than ${process.env.MAX_SIZE_UPLOAD_VIDEO || 2000}MB!`);
      return;
    }

    // setFiles(prevFile => ([...prevFile, {
    //   id: generateUuid(),
    //   name: file.name,
    //   file
    // }]));
    setFiles([{
      id: generateUuid(),
      name: file.name,
      file
    }]);
    onChange(file, []);
  };

  return (
    <div>
      <div className="cursor-pointer relative w-full rounded-md border-dashed border ring-inset ring-gray-300 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:border-transparent focus:ring-orange-500 p-2.5 outline-none sm:text-sm sm:leading-6 border-primary">
        <input
          multiple={multiple}
          type="file"
          className="w-full h-full absolute top-0 left-0 z-1 opacity-0 cursor-pointer"
          accept={process.env.VIDEO_FILE_ACCEPT}
          onChange={handleChange}
        />
        <div className="text-center px-10 py-10">
          <button
            aria-hidden
            type="button"
            className="text-primary"
          >
            <CloudArrowUpIcon width={50} height={50} />
          </button>
          <p className="text-sm">
            Size limit
            :
            {' '}
            {process.env.MAX_SIZE_UPLOAD_VIDEO || 2000}
            MB.
          </p>
          <div>
            {files.length > 0 && files.map((f) => <p className="font-semibold text-lg break-words" key={f.id}>{f.name}</p>)}
          </div>
        </div>
        {progress > 0 && (
        <div className="w-full bg-gray-200 rounded-full dark:bg-gray-700">
          <div className="bg-primary text-xs font-medium text-blue-100 text-center p-0.5 leading-none rounded-full" style={{ width: `${progress}%` }}>{`${Math.floor(progress)}%`}</div>
        </div>
        )}
      </div>
    </div>
  );
}
