'use client';

import { Listbox, ListboxProps } from '@headlessui/react';
import classNames from 'classnames';
import { useEffect, useRef, useState } from 'react';
import { XMarkIcon } from '@heroicons/react/24/solid';
import { categoryService } from '@services/category.services';

interface IProps<T = any> extends ListboxProps<any, T, any> {
  showSearch?: boolean;
  search?: any;
  options: {
    key: string;
    title: string;
  }[];
}

export function Select({
  options,
  placeholder = 'Please select',
  showSearch = false,
  search = null,
  ...props
}: IProps) {
  const inputRef = useRef<HTMLInputElement>(null);
  const ref = useRef<HTMLButtonElement>(null);

  const remove = (tag: string) => {
    props.onChange && props.onChange((props.value || []).filter((key: string) => key !== tag));
    inputRef.current && inputRef.current.focus();
  };

  const render = () => {
    if (!props.multiple) {
      const found = props.value && options.find((option) => option.key === props.value);
      return found ? found.title : placeholder;
    }

    const selected = [] as any[];
    if (Array.isArray(props.value)) {
      props.value.forEach((tag) => {
        const found = options.find((option) => option.key === tag);
        if (found) selected.push(found);
      });
    }
    return selected.map((option) => (
      <div
        key={option.key}
        className="relative text-sm cursor-pointer bg-neutral-100 text-neutral-700 whitespace-nowrap items-center justify-center px-1.5 py-1 pr-5 rounded-md"
      >
        {option.title}
        <XMarkIcon
          className=" absolute right-0 top-1 z-10"
          width={18}
          onClick={(e) => {
            e.preventDefault();
            remove(option.key);
          }}
        />
      </div>
    ));
  };

  const onSearch = async (event: any) => {
    if (event.keyCode === 13) {
      event.preventDefault();
      if (typeof search === 'function') {
        await search(inputRef.current?.value);
      }

      if (inputRef.current) {
        inputRef.current.value = '';
        // inputRef.current.focus();
      }

      ref.current && ref.current.click();
    }
  };

  return (
    <div className="relative">
      <Listbox {...props} as="div">
        <Listbox.Button
          as="div"
          ref={ref}
          onKeyDown={(event) => {
            if (event.keyCode === 13) {
              event.preventDefault();
            }
          }}
          className={classNames(
            'text-left bg-input block w-full rounded-md p-2.5 focus:ring-2 focus:ring-inset focus:ring-orange-500 outline-none  sm:text-sm border border-gray-300 dark:border-zinc-700',
            // selected ? 'text-black dark:text-white' : 'text-gray-500'
            'text-gray-500 flex gap-1.5 flex-wrap'
          )}
        >
          {render()}
          {showSearch && (
            <input
              className="min:w-16 cursor-pointer outline-none flex-1 invisible"
              onKeyDown={onSearch}
              ref={inputRef}
            />
          )}
        </Listbox.Button>
        <Listbox.Options className="absolute top-full left-0  cursor-pointer mt-2 w-56 origin-top-right divide-y divide-gray-100 rounded-md bg-white shadow-lg ring-1 ring-black/5 focus:outline-none transform opacity-100 scale-100 z-10 overflow-auto max-h-[200px]">
          {options.map((option) => (
            <Listbox.Option
              className="text-gray-900 group flex w-full items-center rounded-md px-2 py-2.5 text-sm hover:text-primary'"
              key={option.key}
              value={option.key}
            >
              {({ selected }) => <li className={selected ? 'text-primary' : ''}>{option.title}</li>}
            </Listbox.Option>
          ))}
        </Listbox.Options>
      </Listbox>
    </div>
  );
}

interface TagSelectProps {
  value: string[];
  onChange: (value: string[]) => void;
}

export function TagSelect({ value, onChange }: TagSelectProps) {
  const inputRef = useRef<HTMLInputElement>(null);

  const onKeyDown = (event: any) => {
    if (event.keyCode === 13) {
      event.preventDefault();
      const tag = event.target.value;
      if (tag && value.indexOf(tag) === -1) {
        onChange([...value, tag]);
      }
      if (inputRef.current) {
        inputRef.current.value = '';
        inputRef.current.focus();
      }
    }
  };

  const remove = (tag: string) => {
    onChange(value.filter((key) => key !== tag));
    inputRef.current && inputRef.current.focus();
  };

  return (
    <div className="bg-input color-black block w-full rounded-md p-2.5 focus:ring-2 focus:ring-inset focus:ring-orange-500 outline-none  sm:text-sm sm:leading-6 border border-gray-300 dark:border-zinc-700">
      <div
        className={classNames(
          'flex w-full items-center overflow-auto gap-1.5 flex-wrap'
        )}
      >
        {value.length > 0
          && value.map((tag) => (
            <div
              key={tag}
              className="relative text-sm cursor-pointer bg-neutral-100 text-neutral-700 whitespace-nowrap items-center justify-center px-1.5 py-1 pr-5 rounded-md"
            >
              {tag}
              <XMarkIcon
                className="absolute right-0 top-1 z-10"
                width={18}
                onClick={() => remove(tag)}
              />
            </div>
          ))}
        <input
          className={classNames('cursor-pointer outline-none flex-1 bg-transparent')}
          onKeyDown={onKeyDown}
          ref={inputRef}
        />
      </div>
    </div>
  );
}

export function CategoriesSelect({ ...props }: IProps) {
  const [options, setOptions] = useState<any>([]);

  const search = async (value: string, query = {}) => {
    try {
      const res = await categoryService.search({ q: value, ...query });
      if (res.data) {
        setOptions(
          res.data.data.map((category) => ({
            key: category._id,
            title: category.title
          }))
        );
      }
    } catch (error) { /* empty */ }
  };

  // should remove
  useEffect(() => {
    search('', { limit: 99, group: 'video' });
  }, []);

  return (
    <Select
      {...props}
      placeholder=""
      options={options}
      multiple
      showSearch
      search={search}
    />
  );
}
