'use client';

import { FieldInputProps } from 'formik';
import { MagnifyingGlassIcon } from '@heroicons/react/24/solid';
import { usePathname, useRouter } from 'next/navigation';
import { ReactNode, useRef, useState } from 'react';
import { FaEye, FaEyeSlash } from 'react-icons/fa';
import s from './input.module.scss';

interface IProps<T = any> extends FieldInputProps<T> {
  type?: string;
  placeholder?: string;
  className?: string;
  rows?: number;
  children?: ReactNode;
  id?: string;
}

export function Input({ ...props }: IProps) {
  return (
    <input
      type="text"
      className="bg-input color-black block w-full rounded-md p-2.5 focus:ring-2 focus:ring-inset focus:ring-input outline-none  sm:text-sm sm:leading-6 border border-gray-300 dark:border-zinc-700 dark:focus:border-none"
      {...props}
    />
  );
}

Input.Password = function Password({ ...props }: IProps) {
  const [showPassword, setShowPassword] = useState(false);
  return (
    <>
      <input
        type={showPassword ? 'text' : 'password'}
        className="bg-input color-black block w-full rounded-md p-2.5 focus:ring-2 focus:ring-inset focus:ring-input outline-none  sm:text-sm sm:leading-6 border border-gray-300 dark:border-zinc-700 dark:focus:border-none"
        {...props}
      />
      <span
        role="button"
        tabIndex={0}
        onClick={() => setShowPassword(!showPassword)}
        onKeyDown={(e) => {
          if (e.key === 'Enter' || e.key === ' ') {
            setShowPassword(!showPassword);
          }
        }}
        className="absolute top-3 right-2 cursor-pointer"
      >
        {showPassword ? <FaEye /> : <FaEyeSlash />}
      </span>
    </>
  );
};

Input.TextArea = function TextArea({ ...props }: IProps) {
  return (
    <textarea
      type="text"
      className="resize-none block bg-input w-full rounded-md p-2.5 ring-inset ring-gray-300 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:border-transparent focus:ring-input outline-none  sm:text-sm sm:leading-6  border-gray-300 dark:border-zinc-700 border dark:focus:border-none"
      {...props}
    />
  );
};

Input.Radio = function Radio({ children, id, ...props }: IProps) {
  return (
    // eslint-disable-next-line jsx-a11y/label-has-associated-control
    <label className={s.radio_btn}>
      <input
        type="radio"
        className="bg-input cursor-pointer h-4 w-4 border-gray-300 dark:border-gray-700 text-indigo-600 focus:ring-indigo-600 mr-1"
        id={id}
        {...props}
      />
      <span className={s.checkmark} />
      {children}
    </label>
  );
};

interface InputSearchProps {
  defaultValue?: string;
  pathname?: string;
  onChange?: () => void;
}

export function InputSearch({ defaultValue = '', pathname, onChange }: InputSearchProps) {
  const router = useRouter();
  const path = usePathname();
  const inputRef = useRef<HTMLInputElement>(null);

  const onKeyUp = (event: any) => {
    event.preventDefault();
    if (event.keyCode === 13) {
      const params = new URLSearchParams();
      params.set('q', event.target.value);

      router.push(`${pathname || path}?${params.toString()}`);
      if (typeof onChange === 'function') {
        onChange();
      }
      if (inputRef.current) {
        inputRef.current.value = '';
      }
    }
  };

  return (
    <div className="relative">
      <div className="absolute inset-y-0 end-0 flex items-center pe-3 pointer-events-none">
        <MagnifyingGlassIcon width={23} height={23} />
      </div>
      <input
        type="text"
        ref={inputRef}
        defaultValue={defaultValue}
        onKeyUp={onKeyUp}
        placeholder="Enter Keyword"
        className="bg-input color-black block w-full rounded-md p-2.5 pr-10 focus:ring-2 focus:ring-inset focus:ring-input outline-none  sm:text-sm sm:leading-6 border border-gray-300 dark:border-zinc-700 dark:focus:border-none"
      />
    </div>
  );
}
