'use client';

import { ReactNode, createContext, useContext } from 'react';
import { Pagination } from './pagination';

interface Column {
  key: string;
  dataIndex: string;
  title: ReactNode;
  render?: (field: any, record: any) => ReactNode;
}

interface Props<S = any> {
  total: number;
  paged: number;
  pageSize: number;
  dataSource: S[];
  columns: Column[];
}

const TableContext = createContext<Props>({
  dataSource: [],
  columns: [],
  pageSize: 10,
  total: 0,
  paged: 1
});

function Head() {
  const { columns } = useContext(TableContext);
  return (
    <thead>
      <tr className="dark:bg-input bg-neutral-600 text-white">
        {columns.map((column) => (
          <th className="p-4 text-left font-normal" key={column.key}>{column.title}</th>
        ))}
      </tr>
    </thead>
  );
}

function Cell({ data, dataIndex, render }: any) {
  return <td className="p-4">{render ? render(data[dataIndex], data) : data[dataIndex]}</td>;
}

function Row({ row, ...props }: { row: number }) {
  const { columns, dataSource } = useContext(TableContext);
  return (
    <tr className="p-4 border-b border-neutral-50 dark:border-neutral-600">
      {columns.map((column) => (
        <Cell
          data={dataSource[row]}
          {...props}
          {...column}
          key={`cell-${column.key}`}
        />
      ))}
    </tr>
  );
}

export function Table({ ...props }: Props) {
  return (
    <TableContext.Provider value={props}>
      <div className="relative overflow-x-auto">
        <table className="table-auto w-full text-sm">
          <Head />
          <tbody>
            {props.dataSource.map((data, index: number) => (
              <Row key={data._id} row={index} />
            ))}
          </tbody>
        </table>
      </div>
      <div className="flex items-center">
        <div className="text-sm">
          {props.dataSource.length}
          {' '}
          of
          {' '}
          {props.total}
          {' '}
          items
        </div>
        <div className="ml-auto"><Pagination {...props} page={props.paged} /></div>
      </div>
    </TableContext.Provider>
  );
}
