import cookie from 'js-cookie';
import { Res } from 'src/interfaces';

import { APIRequest } from './api-request';
import { getResponseError } from '@lib/utils';

export class SettingService extends APIRequest {
  public(group = '', headers = {}): Promise<Res<any>> {
    return this.get(this.buildUrl('/settings/public', { group }), headers).catch(async (error) => {
      const e = await Promise.resolve(error);
      return {
        error: getResponseError(e)
      };
    });
  }

  all(group = '', headers = {}): Promise<Res<any>> {
    return this.get(this.buildUrl('/admin/settings', { group }), headers)
      .catch(async (error) => {
        const e = await Promise.resolve(error);
        return {
          error: getResponseError(e)
        };
      });
  }

  valueByKeys(keys: string[]): Promise<Res<any>> {
    return this.post('/settings/keys', { keys }, {}, { next: { tags: ['search', 'setting'] } })
      .catch(async (error) => {
        const e = await Promise.resolve(error);
        return {
          error: getResponseError(e)
        };
      });
  }

  update(key: string, value: any) {
    return this.put(`/admin/settings/${key}`, { value });
  }

  getFileUploadUrl() {
    return this.post('/admin/settings/sign-upload-image-url');
  }

  verifyMailer() {
    return this.post('/mailer/verify');
  }

  uploadFile(formData: FormData) {
    const baseApiEndpoint = this.getBaseApiEndpoint();
    const token = cookie.get('token') || undefined;

    return fetch(`${baseApiEndpoint}/admin/settings/files/upload`, {
      method: 'POST',
      headers: {
        Authorization: token || ''
      },
      body: formData
    })
      .then((response) => {
        if (response.status >= 200 && response.status < 300) {
          return response;
        }
        throw response.clone().json();
      })
      .then((response) => {
        if (response.status === 204 || response.status === 205) {
          return null;
        }
        return response.json();
      });
  }

  public uploadSettingFile(
    uploadUrl: string,
    file: any
  ) {
    return this.upload({
      url: uploadUrl, file, fieldName: 'file'
    });
  }

  getFileUploadedInfo(fileId: string) {
    return this.get(`/admin/settings/files/${fileId}/info`);
  }
}

export const settingService = new SettingService();
