import axios from 'axios';
import cookie from 'js-cookie';

import { isUrl } from '@lib/string';

import { IFileUpload, ISignUploadUrlPayload } from '../interfaces';

export interface IResponse<T> {
  status: number;
  data: T;
}

export const TOKEN = 'token';

export abstract class APIRequest {
  static API_ENDPOINT: string = '';
  getBaseApiEndpoint() {
    return process.env.API_ENDPOINT || process.env.NEXT_PUBLIC_API_ENDPOINT;
  }

  request(
    url: string,
    method?: string,
    body?: any,
    headers?: { [key: string]: string },
    options = {}
  ): Promise<IResponse<any>> {
    const verb = (method || 'get').toUpperCase();

    const updatedHeader = {
      'Content-Type': 'application/json',
      // TODO - check me
      Authorization: cookie.get(TOKEN) || '',
      ...(headers || {})
    };
    const baseApiEndpoint = this.getBaseApiEndpoint();

    return axios({
      method: verb,
      url: isUrl(url) ? url : `${baseApiEndpoint}${url}`,
      data: body ? JSON.stringify(body) : undefined,
      headers: updatedHeader,
      // cache: 'force-cache',
      fetchOptions: {
        cache: 'no-store',
        ...options
      },
      ...options
    })
      .then((resp) => resp.data)
      .catch((e) => {
        const { response } = e;
        if (response.status === 401) {
          if (typeof window !== 'undefined') {
            window.location.href = '/';
          }

          throw new Error('Forbidden in the action!');
        }

        throw response.data;
      });
  }

  buildUrl(baseUrl: string, params?: { [key: string]: any }) {
    if (!params) {
      return baseUrl;
    }

    const queryString = Object.keys(params)
      .map((k) => {
        if (Array.isArray(params[k])) {
          return params[k]
            .map(
              (param: any) =>
                `${encodeURIComponent(k)}=${encodeURIComponent(param)}`
            )
            .join('&');
        }
        return `${encodeURIComponent(k)}=${encodeURIComponent(params[k])}`;
      })
      .join('&');
    return `${baseUrl}?${queryString}`;
  }

  get(url: string, headers?: { [key: string]: any }, options = {}) {
    return this.request(url, 'get', null, headers, options);
  }

  post(
    url: string,
    data?: any,
    headers?: { [key: string]: string },
    options = {}
  ) {
    return this.request(url, 'post', data, headers, options);
  }

  put(url: string, data?: any, headers?: { [key: string]: string }) {
    return this.request(url, 'put', data, headers);
  }

  del(url: string, data?: any, headers?: { [key: string]: string }) {
    return this.request(url, 'delete', data, headers);
  }

  async signUploadUrl({
    url,
    data = {}
  }: ISignUploadUrlPayload) {
    return this.post(url, data);
  }

  async upload({
    url,
    file,
    fieldName = 'file',
    onProgress = () => { },
    method = 'POST',
    fields,
    header = {}
  }: IFileUpload) {
    return new Promise((resolve, reject) => {
      const req = new XMLHttpRequest();

      req.upload.addEventListener('progress', (event) => {
        if (event.lengthComputable) {
          onProgress({
            percentage: (event.loaded / event.total) * 100
          });
        }
      });

      req.addEventListener('load', () => {
        const success = req.status >= 200 && req.status < 300;
        const { response } = req;

        if (!success) {
          return reject(response);
        }
        return resolve({
          url,
          status: success,
          data: response?.data
        });
      });

      req.upload.addEventListener('error', () => {
        reject(req.response);
      });

      const formData = new FormData();
      if (fields && Object.keys(fields).length) {
        Object.keys(fields).forEach((key: any) => {
          if (
            typeof fields[key] !== 'undefined' &&
            !Array.isArray(fields[key])
          )
            formData.append(key, fields[key]);
          if (
            typeof fields[key] !== 'undefined' &&
            Array.isArray(fields[key])
          ) {
            if (fields[key].length) {
              for (
                let i = 0;
                i < fields[key].length;
                i += 1
              ) {
                formData.append(key, fields[key][i]);
              }
            }
          }
        });
      }
      // append file at last is required for AWS upload
      formData.append(fieldName, file, file.name);
      req.responseType = 'json';
      req.open(method || 'POST', url);
      if (header && Object.keys(header).length) {
        Object.keys(header).forEach((key: any) => {
          req.setRequestHeader(key, header[key]);
        });
      }
      req.send(formData);
    });
  }
}
