'use client';

import { ICategory, IPerformer, IVideo } from 'src/interfaces';
import { getResponseError } from 'src/lib';
import { videoService } from 'src/services';
import { Button, Space, Table, TablePaginationConfig, Tag, message } from 'antd';
import Link from 'next/link';
import { useRouter, useSearchParams } from 'next/navigation';
import StatusBadge from '@components/common/status-badge';
import { formatDate } from '@lib/date';
import { FileStatus } from '@components/file/file-status';
import { useState } from 'react';
import MediaPreviewModal from '@components/media/preview-modal';

type Props = {
  data: IVideo[];
  total: number;
  pageSize: number;
};

export function VideoTableList({ data, total, pageSize }: Props) {
  const [selectedRowKeys, setSelectedRowKeys] = useState<React.Key[]>([]);
  const [loading, setLoading] = useState(false);
  const [updating, setUpdating] = useState(false);
  const router = useRouter();

  const searchParams = useSearchParams();
  const page = searchParams.get('page');


  const handleChange = (pagination: TablePaginationConfig) => {
    const currentPage = pagination.current && !isNaN(pagination.current) ? pagination.current : 1;
    const current = new URLSearchParams(Array.from(searchParams.entries()));
    current.set('page', `${currentPage}`);
    current.set('pageSize', `${pagination.pageSize}`);
    router.push(`/video?${current.toString()}`);
  };

  const handleRemove = async (id: string) => {
    try {
      if (!window.confirm('Are you sure you want to delete this Video?')) return;
      await videoService.delete(id);
      message.success('Deleted Successfully.');
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    }
  };

  const removeALl = async () => {
    // ajax request after empty completing
    try {
      if (!window.confirm('Are you sure you want to delete all selected Videos?')) return;
      setLoading(true);
      await selectedRowKeys.reduce(async (lp, id) => {
        await lp;
        await videoService.delete(id as string);
        return Promise.resolve();
      }, Promise.resolve());
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setLoading(false);
      setSelectedRowKeys([]);
    }
  };

  const onSelectChange = (newSelectedRowKeys: React.Key[]) => {
    setSelectedRowKeys(newSelectedRowKeys);
  };

  const handleUpdate = async (id: string, field: string, value: any) => {
    try {
      setUpdating(true);
      await videoService.update(id, { [field]: value });
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setUpdating(false)
    }
  }

  const rowSelection = {
    selectedRowKeys,
    onChange: onSelectChange
  };
  const hasSelected = selectedRowKeys.length > 0;

  const dataSource = data.map(d => ({
    ...d,
    key: d._id
  })) as IVideo[];


  const columns = [
    {
      title: 'Thumbnail',
      key: 'thumbnail',
      render: (record: IVideo) => {
        const { convertStatus, thumbnails, video, _id, teaser, embeddedCode } = record;
        if (convertStatus === 'error') {
          return <img src={'/video-convert-fail.png'} alt="video thumbnail" width={50} height={50} />
        }
        if (convertStatus === 'processed') {
          if (embeddedCode) {
            return (
              <Space>
                {thumbnails && thumbnails.length && <img src={thumbnails[0]} alt="video thumbnail" width={50} height={50} />}
                <MediaPreviewModal key={_id} mediaType="embeddedCode" url={embeddedCode} />
              </Space>
            )
          }
          return (
            <Space>
              {thumbnails && thumbnails.length && <img src={thumbnails[0]} alt="video thumbnail" width={50} height={50} />}
              {video && video.url ?
                <MediaPreviewModal key={_id} mediaType="video" url={video.url} /> :
                <MediaPreviewModal key={_id} mediaType="video" url={teaser?.toString() || ''} />
              }
            </Space>
          )
        }
        return <img src={'/no-image.jpg'} alt="video thumbnail" width={50} height={50} />
      },
      width: 250
    },
    {
      title: 'Title',
      dataIndex: 'title',
      key: 'title',
      ellipsis: true,
      width: 300
    },
    {
      title: 'Pornstar',
      dataIndex: 'performers',
      key: 'performers',
      render: (performers: IPerformer[]) => {
        const performerName = performers?.length ? performers.map(p => p.name || p.username) : []
        return <span>{performerName.length ? performerName.join(', ') : 'N/A'}</span>
      },
      width: 250,
      ellipsis: true
    },
    {
      title: 'Category',
      dataIndex: 'categories',
      key: 'categories',
      render: (categories: ICategory[], record: any) => {
        if (record.source === 'fapcash') {
          return categories?.map((category, idx) => (
            <div key={idx}>
              {category.title.split(';').length > 2 ? (
                category.title.split(';').slice(0, 2).map((item, index) => (
                  <Tag color="blue" key={index}>
                    {item.trim()}
                  </Tag>
                ))
              ) : (
                category.title.split(';').map((item, index) => (
                  <Tag color="blue" key={index}>
                    {item.trim()}
                  </Tag>
                ))
              )}
            </div>
          ));
        } else {
          const categoryName = categories?.length ? categories.map(c => c.title || '') : [];
          return <span>{categoryName.length ? categoryName.join(', ') : 'N/A'}</span>;
        }
      },
      ellipsis: true,
      width: 200
    },
    {
      title: 'Convert Status',
      dataIndex: 'convertStatus',
      key: 'convertStatus',
      render: (convertStatus: string) => <FileStatus status={convertStatus} />,
      width: 100
    },
    {
      title: 'Status',
      dataIndex: 'status',
      key: 'status',
      render: (status: string, record: IVideo) => {
        const { _id } = record;
        const isActive = status === 'active';
        const statusUpdate = isActive ? 'inactive' : 'active';
        return (<span style={{ cursor: 'pointer' }} onClick={() => handleUpdate(_id, 'status', statusUpdate)}><StatusBadge active={isActive} activeText="Active" inActiveText="Inactive" /></span>)
      },
      width: 100
    },
    {
      title: 'Updated At',
      dataIndex: 'updatedAt',
      key: 'updatedAt',
      render(date: Date) {
        return <span>{formatDate(date)}</span>;
      },
      width: 250
    },
    {
      title: 'Action',
      key: 'action',
      render: (record: any) => (
        <Space>
          <Link href={`/video/update/${record._id}`}>
            <Button type="primary">Update</Button>
          </Link>
          <Button onClick={() => handleRemove(record._id)}>Delete</Button>
        </Space>
      ),
      width: 200
    }
  ];

  return (
    <div>
      <div style={{ marginBottom: 16 }}>
        <Button type="primary" danger onClick={removeALl} disabled={!hasSelected} loading={loading}>
          Delete Videos
        </Button>
        <span style={{ marginLeft: 8 }}>
          {hasSelected ? `Selected ${selectedRowKeys.length} items` : ''}
        </span>
      </div>
      <Table
        className="ant-border-space"
        columns={columns}
        dataSource={dataSource}
        pagination={{
          pageSize: pageSize,
          total: total,
          current: page ? Number(page) : 1,
          showSizeChanger: true
        }}
        onChange={handleChange}
        scroll={{ x: 1200 }}
        rowSelection={rowSelection}
        loading={loading || updating}
      />
    </div>
  );
}
