'use client';

import {
  Form,
  Button,
  Select,
  Input,
  message,
  Checkbox,
  Divider
} from 'antd';
import { FormInstance } from 'antd/lib/form';
import { forwardRef, useEffect, useImperativeHandle, useRef, useState } from 'react';
import {
  useFormStatus,
  useFormState
} from 'react-dom';
import { ActionState, IVideo, IVideoUpload } from 'src/interfaces';
import { videoService } from '@services/video.service';
import MediaPreviewModal from '@components/media/preview-modal';
import PerformerSelection from '@components/performer/performer-selection';
import TagSelection from '@components/tag/tag-selection';
import CategorySelection from '@components/category/category-selection';
import FileUploadCustom from '@components/file/file-upload-custom';
import { RcFile, UploadFile } from 'antd/lib/upload';
import { STATUS } from 'src/constants';
import { redirect } from 'next/navigation';

interface IProps {
  data?: IVideo | undefined;
  onFinish: any;
  files?: UploadFile[];
  canSubmit?: boolean;
  onSubmitFailed?: Function;
}

function FormVideo({ data = undefined, onFinish, files = [], canSubmit = true, onSubmitFailed }: IProps, ref: any) {
  const [state, formAction] = useFormState<ActionState, IVideoUpload>(onFinish, {});
  const formRef = useRef<FormInstance>(null);
  const { pending } = useFormStatus();
  const [fileList, setFileList] = useState<UploadFile[]>(files);
  const [uploading, setUploading] = useState(false);

  function setFormVal(field: string, val: any) {
    const instance = formRef.current as FormInstance;
    instance.setFieldsValue({
      [field]: val
    });
  }
  const generateSlug = (value:string) =>{
    const slugValue = value.toLowerCase().trim().split('').map((char)=>{
      return /^[a-z0-9]+((-|_)[a-z0-9]+)*$/.test(char) ? char : '-';
    }).join('').replace(/--+/g,'-');
    const instance = formRef.current as FormInstance;
    instance.setFieldValue('slug', slugValue);
  }

  useEffect(() => {
    if (state.success || state.error) {
      setUploading(false);
    }
    if (state.success) {
      message.success('Updated Successfully');
      redirect('/video');
    } else {
      state.message && message.error(state.message);
    }
  }, [state]);

  useEffect(() => {
    if (data) {
      formRef.current?.setFieldsValue(data);
    }
  }, [data]);

  const upload = async (file: UploadFile): Promise<any> => {
    const { data: signUrlResp } = await videoService.signUploadFileUrl();
    if (signUrlResp.uploadUrl) {
      const resp = await videoService.uploadVideoFile(signUrlResp.uploadUrl, file, ({ percentage }: any) => {
        // setFileResponse({ [file.uid]: { status: percentage < 100 ? 'uploading' : 'done', percent: percentage } })
        const index = fileList.indexOf(file);
        if (index > -1) {
          setFileList([
            ...fileList.slice(0, index),
            {
              ...fileList[index],
              name: file.name,
              status: percentage < 100 ? 'uploading' : 'done',
              percent: percentage,
              type: file.type
            },
            ...fileList.slice(2)
          ]);
        }

      }) as any;
      if (resp.status) {
        const { data: fileInfo } = await videoService.getFileUploadedInfo(signUrlResp.fileId);
        return {
          fileId: fileInfo._id
        }
      }
      return null;
    }
    return null
  }

  const beforeUpload = (file: RcFile) => {
    const isLt2M = file.size / 1024 / 1024 < (Number(process.env.MAX_VIDEO_SIZE_FILE || 2048));
    if (!isLt2M) {
      message.error(`File must smaller than ${process.env.MAX_VIDEO_SIZE_FILE || 2048}MB!`);
      return false;
    }
    setFileList([file]);
    return false;
  }

  const onRemove = (file: UploadFile) => {
    const index = fileList.indexOf(file);
    const newFileList = fileList.slice();
    newFileList.splice(index, 1);
    setFileList(newFileList);
  }

  const submit = async (values: Record<string, any>) => {
    if (!fileList.length && !data) {
      return message.error('Please select a video');
    }
    setUploading(true);
    const formData = { ...values } as IVideoUpload;
    if (fileList.length > 0 && fileList[0].status !== 'done') {
      const res = await upload(fileList[0]);
      if (!res || !res.fileId) {
        return message.error('Upload video failed');
      }
      formData.fileId = res.fileId;
    }
    if (canSubmit) {
      return formAction(formData);
    }
    onFinish(formData);
  }

  useImperativeHandle(ref, () => ({
    submit: () => formRef.current?.submit(),
    validate: () => formRef.current?.validateFields()
  }));

  return (
    <Form
      ref={formRef}
      onFinish={(values) => submit({ ...data, ...values })}
      initialValues={{
        title: '',
        description: '',
        slug: '',
        seoTitle: '',
        seoKeywords: '',
        seoDesc: '',
        status: STATUS.ACTIVE,
        verified: true
      }}
      onFinishFailed={() => onSubmitFailed && onSubmitFailed()}
      layout="vertical"
    >
      <Form.Item
        name="title"
        label="Title"
        rules={[{ required: true, message: 'Required!' }]}
      >
        <Input
          onChange={(e) => generateSlug(e.target.value)}
          placeholder="Enter title"
        />
      </Form.Item>
      <Form.Item
        name="slug"
        label="Slug"
      >
        <Input placeholder="Enter a Custom URL" />
      </Form.Item>
      <Form.Item name="description" label="Description">
        <Input.TextArea rows={5} placeholder="Enter description" />
      </Form.Item>
      <Form.Item name="seoTitle" label="Title (SEO)">
        <Input placeholder="Enter seo title" />
      </Form.Item>
      <Form.Item name="seoKeywords" label="Keyword (SEO)">
        <Input placeholder="Enter keywords" />
      </Form.Item>
      <Form.Item name="seoDesc" label="Description (SEO)">
        <Input placeholder="Enter SEO description" />
      </Form.Item>
      <Form.Item name="performerIds" label="Performers">
        <PerformerSelection
          selectProps={{
            mode: 'multiple',
            defaultValue: (data && data.performerIds) || []
          }}
          onChange={(field: string, value: any) => setFormVal(field, value)}
          fieldName="performerIds"
        />
      </Form.Item>
      <Form.Item name="tagIds" label="Tags">
        <TagSelection
          selectProps={{
            mode: 'tags',
            defaultValue: (data && data.tags) || []
          }}
          onChange={(field: string, value: any) => setFormVal(field, value)}
          fieldName="tagIds"
        />
      </Form.Item>
      <Form.Item name="categoryIds" label="Categories">
        <CategorySelection
          selectProps={{
            mode: 'multiple',
            defaultValue: (data && data.categoryIds) || []
          }}
          onChange={(field: string, value: any) => setFormVal(field, value)}
          fieldName="categoryIds"
          group="video"
        />
      </Form.Item>
      <Form.Item
        name="status"
        label="Status"
        rules={[{ required: true, message: 'Please select status!' }]}
      >
        <Select placeholder={'Select status'}>
          <Select.Option key="active" value="active">
            Active
          </Select.Option>
          <Select.Option key="inactive" value="inactive">
            Inactive
          </Select.Option>
        </Select>
      </Form.Item>
      {/* <Form.Item name="verified" valuePropName="checked">
        <Checkbox>Verified?</Checkbox>
      </Form.Item> */}
      <Form.Item name="featured" valuePropName="checked" hidden>
        <Checkbox>Is this a Featured Video?</Checkbox>
      </Form.Item>
      {(!data || data?.source === 'internal') && (
        <Form.Item
          label="Upload video"
          extra={
            <div>
              <p>Maximum file size up to 2GB.</p>
              <p>Accept video/*,.mkv,.flv</p>
            </div>
          }
        >
          {data && data.video && data.video.url && (
            <>
              <MediaPreviewModal
                key={data._id}
                mediaType="video"
                url={data.video.url}
              />
              <Divider />
            </>
          )}

          <FileUploadCustom
            uploadProps={{
              name: 'video-upload',
              accept: process.env.VIDEO_FILE_ACCEPT,
              showUploadList: true,
              listType: 'picture',
              beforeUpload,
              onRemove: (file) => onRemove(file),
              fileList
            }}
            uploading={uploading}
          />
        </Form.Item>
      )}
      {canSubmit && (
        <Form.Item>
          <Button
            type="primary"
            htmlType="submit"
            loading={pending}
            disabled={pending}
          >
            Submit
          </Button>
        </Form.Item>
      )}
    </Form>
  );
}
export default forwardRef(FormVideo);