'use client';

import { ITag } from 'src/interfaces';
import { getResponseError } from 'src/lib';
import { tagService } from 'src/services';
import { Button, Space, Table, TablePaginationConfig, message } from 'antd';
import Link from 'next/link';
import { useRouter, useSearchParams } from 'next/navigation';
import { formatDate } from '@lib/date';
import { useState } from 'react';
import { SorterResult } from 'antd/es/table/interface';
import { TableProps } from 'antd/lib';
import StatusBadge from '@components/common/status-badge';

type Props = {
  data: ITag[];
  total: number;
  pageSize: number;
};

export function TagTableList({ data, total, pageSize }: Props) {
  const [selectedRowKeys, setSelectedRowKeys] = useState<React.Key[]>([]);
  const [loading, setLoading] = useState(false);
  const [updating, setUpdating] = useState(false);
  const router = useRouter();

  const searchParams = useSearchParams();
  const page = searchParams.get('page');

  const handleChange: TableProps<ITag>['onChange'] = (pagination: TablePaginationConfig, filters, sorter) => {
    const { field, order } = sorter as SorterResult<ITag>;
    const sort = order ? (order === 'descend' ? 'desc' : 'asc') : '';
    let sortBy = '';
    if (field) {
      switch (field) {
        case 'tagKey':
          sortBy = 'key'
          break;
        default:
          sortBy = field.toString();
          break;
      }
    }
    const current = new URLSearchParams(Array.from(searchParams.entries()));
    current.set('page', `${pagination.current || 1}`);
    current.set('sort', `${sort}`);
    current.set('sortBy', `${sortBy}`);
    router.push(`/tag?${current.toString()}`);
  };

  const handleRemove = async (id: string) => {
    try {
      if (!window.confirm('Are you sure you want to delete this Tag?')) return;
      await tagService.delete(id);
      message.success('Deleted Successfully.');
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    }
  };

  const removeALl = async () => {
    // ajax request after empty completing
    try {
      if (!window.confirm('Are you sure you want to delete all selected Ads?')) return;
      setLoading(true);
      await selectedRowKeys.reduce(async (lp, id) => {
        await lp;
        await tagService.delete(id as string);
        return Promise.resolve();
      }, Promise.resolve());
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setLoading(false);
      setSelectedRowKeys([]);
    }
  };

  const onSelectChange = (newSelectedRowKeys: React.Key[]) => {
    setSelectedRowKeys(newSelectedRowKeys);
  };

  const handleUpdate = async (id: string, field: string, value: any) => {
    try {
      setUpdating(true);
      await tagService.update(id, { [field]: value });
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setUpdating(false);
    }
  }

  const rowSelection = {
    selectedRowKeys,
    onChange: onSelectChange
  };
  const hasSelected = selectedRowKeys.length > 0;

  const dataSource = data.map(d => ({
    ...d,
    key: d._id,
    tagKey: d.key
  })) as ITag[];

  const columns = [
    {
      title: 'Name',
      dataIndex: 'name',
      key: 'name',
      sorter: true
    },
    {
      title: 'Key',
      dataIndex: 'tagKey',
      key: 'tagKey',
      sorter: true
    },
    {
      title: 'Status',
      dataIndex: 'status',
      key: 'status',
      render: (status: string, record: ITag) => {
        const { _id } = record;
        const isActive = status === 'active';
        const statusUpdate = isActive ? 'inactive' : 'active';
        return (<span style={{ cursor: 'pointer' }} onClick={() => handleUpdate(_id, 'status', statusUpdate)}><StatusBadge active={isActive} activeText="Active" inActiveText="Inactive" /></span>)
      },
      sorter: true
    },
    {
      title: 'Updated At',
      dataIndex: 'updatedAt',
      key: 'updatedAt',
      render(date: Date) {
        return <span>{formatDate(date)}</span>;
      },
      sorter: true
    },
    {
      title: 'Action',
      key: 'action',
      render: (record: any) => (
        <Space>
          <Link href={`/tag/update/${record._id}`}>
            <Button type="primary">Update</Button>
          </Link>
          <Button onClick={() => handleRemove(record._id)}>Delete</Button>
        </Space>
      )
    }
  ];

  return (
    <div>
      <div style={{ marginBottom: 16 }}>
        <Button type="primary" danger onClick={removeALl} disabled={!hasSelected} loading={loading}>
          Delete Tags
        </Button>
        <span style={{ marginLeft: 8 }}>
          {hasSelected ? `Selected ${selectedRowKeys.length} items` : ''}
        </span>
      </div>
      <Table
        className="ant-border-space"
        columns={columns}
        dataSource={dataSource}
        pagination={{
          pageSize: pageSize,
          total: total,
          current: page ? Number(page) : 1
        }}
        onChange={handleChange}
        scroll={{ x: 1200 }}
        rowSelection={rowSelection}
        loading={loading || updating}
      />
    </div>
  );
}
