'use client';

import { Form, Input, InputNumber, Radio, Select, Switch } from 'antd';
import { ISetting } from 'src/interfaces';
import SettingFormUpload from './setting-form-upload';
import { useRef, useState } from 'react';
import dynamic from 'next/dynamic';

const Editor = dynamic(() => import('@components/common/suneditor'), {
  ssr: false
});

interface IProps {
  setting: ISetting;
  updating?: boolean;
  onChange: Function;
  onUploaded?: Function;
}

export default function SettingFormItem({ setting, onChange, onUploaded = () => { } }: IProps) {
  const [hasErrors, setHasErrors] = useState<string[]>([]);
  const inputRef = useRef<any>();
  const { type, key, value } = setting;

  const handleUploaded = (field: string, val: any) => {
    // eslint-disable-next-line no-param-reassign
    inputRef.current.input.value = val;
    onUploaded(field, val)
  }

  switch (type) {
    case 'textarea':
      return (
        <Form.Item label={setting.name} key={setting._id} help={setting.description} extra={setting.extra}>
          <Input.TextArea rows={5} defaultValue={value} onChange={(val) => onChange(key, val.target.value)} />
        </Form.Item>
      );
    case 'number':
      return (
        <Form.Item
          name={setting.key}
          label={setting.name}
          key={setting._id}
          extra={setting.extra}
          validateStatus={hasErrors.includes(key) ? 'error' : 'validating'}
          help={hasErrors.includes(key) ? 'Value is not valid' : setting.description}
          rules={[
            ({ }) => ({
              validator(_, val) {
                if (val === null || val < 0) {
                  setHasErrors([...hasErrors, key])
                  return Promise.reject();
                }
                const newHasErrors = hasErrors.filter(eKey => eKey !== key);
                setHasErrors(newHasErrors);
                return Promise.resolve();
              }
            })
          ]}
        >
          <InputNumber
            style={{ width: '100%' }}
            defaultValue={value}
            onChange={(val) => onChange(key, (val || 0))}
            min={(setting.meta && typeof setting.meta.min === 'number') ? setting.meta.min : Number.MIN_SAFE_INTEGER}
            max={(setting.meta && typeof setting.meta.max === 'number') ? setting.meta.max : Number.MAX_SAFE_INTEGER}
            step={(setting.meta && typeof setting.meta.step === 'number') ? setting.meta.step : 1}
          />
        </Form.Item>
      );
    case 'boolean':
      return (
        <Form.Item label={setting.name} key={setting._id} help={setting.description} valuePropName="checked">
          <Switch defaultChecked={value} onChange={(val) => onChange(key, val)} />
        </Form.Item>
      );
    case 'radio':
      return (
        <Form.Item label={setting.name} key={setting._id} help={setting.description} extra={setting.extra}>
          <Radio.Group onChange={(val) => onChange(key, val.target.value)} defaultValue={value}>
            {setting.meta?.value.map((v: any) => (
              <Radio value={v.key} key={v.key}>
                {v.name}
              </Radio>
            ))}
          </Radio.Group>
        </Form.Item>
      );
    case 'select':
      const meta = setting.meta;
      if (!meta.options || !meta.options.length) {
        return null;
      }
      return (
        <Form.Item label={setting.name} key={setting._id} help={setting.description} extra={setting.extra}>
          <Select defaultValue={value} options={meta.options} mode={meta.mode} onChange={(val) => onChange(key, val)} />
        </Form.Item>
      );
      case 'text-editor':
        return (
          <Form.Item
            label={setting.name}
            key={setting._id}
            help={setting.description}
          >
            <Editor
               richTextEditorHtml={setting.value || ''}
               onRichTextEditorChange={({ html }: any) => onChange(setting.key, html)}
               displayMode="EDIT"
            />
          </Form.Item>
        );
    default:
      return (
        <Form.Item label={setting.name} key={setting._id} help={setting.description} extra={setting.extra}>
          <Input
            defaultValue={value}
            key={`input${setting._id}`}
            onChange={(val) => onChange(key, val.target.value)}
            ref={inputRef}
          />
          <SettingFormUpload setting={setting} onUploaded={handleUploaded} />
        </Form.Item>
      );
  }
}