'use client';
import StatusBadge from '@components/common/status-badge';
import { formatDate } from '@lib/date';
import { getResponseError } from '@lib/utils';
import { profileService } from '@services/profile.service';
import {
  Button, Space, Table, TablePaginationConfig, message
} from 'antd';
import { SorterResult } from 'antd/es/table/interface';
import { TableProps } from 'antd/lib';
import Link from 'next/link';
import { useRouter, useSearchParams } from 'next/navigation';
import { useState } from 'react';
import { IProfile } from 'src/interfaces';



export default function ProfileTable({ pageSize, data, total }: any) {
  const router = useRouter();
  const searchParams = useSearchParams();
  const page = searchParams.get('page');
  const [loading, setLoading] = useState(false);

  const handleChange: TableProps<IProfile>['onChange'] = (pagination: TablePaginationConfig, filters, sorter) => {
    const { field = '', order } = sorter as SorterResult<IProfile>;
    const sort = order ? (order === 'descend' ? 'desc' : 'asc') : '';
    const current = new URLSearchParams(Array.from(searchParams.entries()));
    current.set('page', `${pagination.current || 1}`);
    current.set('sort', `${sort}`);
    current.set('sortBy', `${field}`);
    router.push(`/user?${current.toString()}`);
  };

  const handleUpdate = async (id: string, field: string, value: any) => {
    try {
      setLoading(true);
      await profileService.update(id, { [field]: value });
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setLoading(false)
    }
  }
  // table code start
  const columns = [
    {
      title: 'Display name',
      dataIndex: 'name',
      key: 'name',
      sorter: true
    },
    {
      title: 'Username',
      dataIndex: 'username',
      key: 'username',
      sorter: true
    },
    {
      title: 'Email address',
      dataIndex: 'email',
      key: 'email',
      sorter: true
    },
    {
      title: 'Status',
      render(profile: IProfile) {
        const { _id, status } = profile;
        const isActive = status === 'active';
        const statusUpdate = isActive ? 'inactive' : 'active';
        return (<span style={{ cursor: 'pointer' }} onClick={() => handleUpdate(_id, 'status', statusUpdate)}><StatusBadge active={isActive} activeText="Active" inActiveText="Inactive" /></span>)
      }
    },
    {
      title: 'Updated At',
      dataIndex: 'updatedAt',
      key: 'updatedAt',
      render(date: Date) {
        return <span>{formatDate(date)}</span>;
      },
      sorter: true
    },
    {
      title: 'Action',
      dataIndex: '_id',
      render(id: string) {
        return (
          <Space>
            <Link href={{
              pathname: '/user/update',
              query: { id }
            }}>
              <Button type="primary">Update</Button>
            </Link>
          </Space>
        );
      }
    }
  ];

  return (
    <Table
      pagination={{
        pageSize: pageSize,
        total: total,
        current: page ? Number(page) : 1
      }}
      className="ant-border-space"
      onChange={handleChange}
      columns={columns}
      dataSource={data}
      loading={loading}
    />
  );
}
