'use client';

import {
  Button, Col, DatePicker, Form, Input,
  Row, Select, Switch, message
} from 'antd';
import { useRouter } from 'next/navigation';

import { IProfile } from 'src/interfaces';
import { getResponseError } from 'src/lib';
import { profileService } from 'src/services';

import { AvatarUpdate } from './avatar-update';
import s from './profile-form.module.scss';
import dayjs from 'dayjs';
import { STATUS } from 'src/constants';
import { useRef } from 'react';

const validateMessages = {
  required: 'This field is required!',
  types: {
    email: 'Not a valid email!',
    number: 'Not a valid number!'
  },
  number: {
    // eslint-disable-next-line no-template-curly-in-string
    range: 'Must be between ${min} and ${max}'
  }
};

type IProps = {
  profile?: IProfile;
};

export function ProfileForm({
  profile = undefined
}: IProps) {
  const fileId = useRef<string>();
  const router = useRouter();

  const [form] = Form.useForm();

  const create = async (data: any) => {
    try {
      if (fileId.current) {
        data.avatarId = fileId.current;
      }
      const resp = await profileService.create(data);
      message.success('Created successfully');
      router.push(`/user/update?id=${resp.data._id}`);
    } catch (e) {
      const err = await Promise.resolve(e);
      message.error(getResponseError(err) || 'An error occurred, please try again!');
    }
  };


  const update = async (data: any) => {
    try {
      if (fileId.current) {
        data.avatarId = fileId.current;
      }
      await profileService.update(profile!._id, data);
      message.success('Updated successfully');
      router.push('/user');
    } catch (e) {
      const err = await Promise.resolve(e);
      message.error(getResponseError(err) || 'An error occurred, please try again!');
    } finally {
      router.refresh();
    }
  };

  const handleAvatarChange = (id: string) => {
    fileId.current = id;
  }

  return (

    <Form
      name="nest-messages"
      labelCol={{ span: 24 }}
      wrapperCol={{ span: 24 }}
      onFinish={profile ? update : create}
      validateMessages={validateMessages}
      form={form}
      initialValues={
        (profile && {
          ...profile,
          dateOfBirth: profile?.dateOfBirth ? dayjs(profile?.dateOfBirth) : dayjs(new Date(2000, 1, 1))
        }) || {
          country: 'US',
          status: STATUS.ACTIVE,
          gender: 'male',
          roles: ['user'],
          verifiedEmail: false
        }
      }
    >
      <Row gutter={12}>
        <Col xs={24} md={12}>
          <Form.Item
            name="firstName"
            label="First Name"
            validateTrigger={['onChange', 'onBlur']}
            rules={[
              // { required: true, message: 'Please input your first name!' },
              {
                pattern: /^[a-zA-ZàáâäãåąčćęèéêëėįìíîïłńòóôöõøùúûüųūÿýżźñçčšžÀÁÂÄÃÅĄĆČĖĘÈÉÊËÌÍÎÏĮŁŃÒÓÔÖÕØÙÚÛÜŲŪŸÝŻŹÑßÇŒÆČŠŽ∂ð ,.'-]+$/u,
                message:
                  'First name can not contain number and special character'
              }
            ]}
          >
            <Input />
          </Form.Item>
        </Col>
        <Col xs={24} md={12}>
          <Form.Item
            name="lastName"
            label="Last Name"
            validateTrigger={['onChange', 'onBlur']}
            rules={[
              // { required: true, message: 'Please input your last name!' },
              {
                pattern: /^[a-zA-ZàáâäãåąčćęèéêëėįìíîïłńòóôöõøùúûüųūÿýżźñçčšžÀÁÂÄÃÅĄĆČĖĘÈÉÊËÌÍÎÏĮŁŃÒÓÔÖÕØÙÚÛÜŲŪŸÝŻŹÑßÇŒÆČŠŽ∂ð ,.'-]+$/u,
                message:
                  'Last name can not contain number and special character'
              }
            ]}
          >
            <Input />
          </Form.Item>
        </Col>
        <Col xs={24} md={12}>
          <Form.Item
            name="username"
            label="Username"
            rules={[
              { required: true, message: 'Please enter username' },
              {
                pattern: /^[a-zA-Z0-9]+$/g,
                message: 'Username must contain only alphanumeric characters'
              }, { min: 3 }]}
          >
            <Input placeholder="Unique, lowercase alphanumeric characters" />
          </Form.Item>
        </Col>
        <Col xs={24} md={12}>
          <Form.Item
            name="name"
            label="Display name"
            validateTrigger={['onChange', 'onBlur']}
            rules={[
              // { required: true, message: 'Please input your display name!' },
              {
                pattern: /^(?=.*\S).+$/g,
                message:
                  'Display name can not contain only whitespace'
              },
              {
                min: 3,
                message: 'Display name must containt at least 3 characters'
              }
            ]}
          >
            <Input />
          </Form.Item>
        </Col>
        <Col xs={24} md={12}>
          <Form.Item name="email" label="Email" rules={[{ type: 'email', required: true, message: 'Please enter email!' }]}>
            <Input />
          </Form.Item>
        </Col>
        <Col xs={24} md={12}>
          <Form.Item name="dateOfBirth" label="Date Of Birth" rules={[
            {
              required: true,
              message: 'Select date of birth'
            }
          ]}>
            <DatePicker placeholder="DD/MM/YYYY"
              format="DD/MM/YYYY"
              disabledDate={(currentDate) => currentDate
                && currentDate > dayjs().subtract(18, 'year').endOf('day')} />
          </Form.Item>
        </Col>
        {!profile && [
          <Col md={12} xs={24} key="password">
            <Form.Item
              label="Password"
              name="password"
              rules={[
                { required: true, message: 'Please input your password!' },
                {
                  min: 8,
                  message: 'Password must have minimum 8 characters'
                }
              ]}
            >
              <Input.Password placeholder="Password" />
            </Form.Item>
          </Col>,
          <Col md={12} xs={24} key="confirmPasswrod">
            <Form.Item
              label="Confirm Password"
              name="confirmPasswrod"
              validateTrigger={['onChange', 'onBlur']}
              dependencies={['password']}
              hasFeedback
              rules={[
                {
                  required: true,
                  message: 'Please confirm your password!'
                },
                ({ getFieldValue }) => ({
                  validator(rule, value) {
                    if (!value || getFieldValue('password') === value) {
                      return Promise.resolve();
                    }
                    // eslint-disable-next-line prefer-promise-reject-errors
                    return Promise.reject('Passwords do not match together!');
                  }
                })
              ]}
            >
              <Input.Password placeholder="Confirm password" />
            </Form.Item>
          </Col>
        ]}
        <Col xs={24} md={12}>
          <Form.Item name="gender" label="Gender" rules={[{ required: true }]}>
            <Select>
              <Select.Option key="male" value="male">
                Male
              </Select.Option>
              <Select.Option key="female" value="female">
                Female
              </Select.Option>
              <Select.Option key="transgender" value="transgender">
                Transgender
              </Select.Option>
            </Select>
          </Form.Item>
        </Col>
        <Col xs={24} md={12}>
          <Form.Item name="status" label="Status">
            <Select>
              <Select.Option key="active" value="active">
                Active
              </Select.Option>
              <Select.Option key="inactive" value="inactive">
                Inactive
              </Select.Option>
            </Select>
          </Form.Item>
        </Col>
        <Col xs={24} md={24}>
          <Form.Item name="verifiedEmail" label="Verified Email?" valuePropName="checked" help="Turn on if email account verified">
            <Switch />
          </Form.Item>
        </Col>
        <Col xs={24} md={24}>
          <Form.Item
            label="Avatar"
            name="avatar"
            extra={<div>
              <p>Maximum file size up to {process.env.MAX_SIZE_IMAGE || 5}MB.</p>
              <p>Accept image/*</p>
              <p>Recommended image size 250x250 pixels</p>
            </div>
            }
          >
            <AvatarUpdate onChange={handleAvatarChange} profile={profile} />
          </Form.Item>
        </Col>
      </Row>
      <Form.Item className={s['text-center']}>
        <Button type="primary" htmlType="submit" loading={false}>
          Submit
        </Button>
      </Form.Item>
    </Form>
  );
}
