'use client';

import { IPost } from 'src/interfaces';
import { getResponseError } from 'src/lib';
import { postService } from 'src/services';
import { Button, Space, Table, TablePaginationConfig, message } from 'antd';
import Link from 'next/link';
import { useRouter, useSearchParams } from 'next/navigation';
import StatusBadge from '@components/common/status-badge';
import { formatDate } from '@lib/date';
import { TableProps } from 'antd/lib';
import { SorterResult } from 'antd/es/table/interface';
import { useState } from 'react';

type Props = {
  data: IPost[];
  total: number;
  pageSize: number;
};

export function PostTableList({ data, total, pageSize }: Props) {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const searchParams = useSearchParams();
  const page = searchParams.get('page');

  const handleChange: TableProps<IPost>['onChange'] = (pagination: TablePaginationConfig, filters, sorter) => {
    const { field = '', order } = sorter as SorterResult<IPost>;
    const sort = order ? (order === 'descend' ? 'desc' : 'asc') : '';
    const current = new URLSearchParams(Array.from(searchParams.entries()));
    current.set('page', `${pagination.current || 1}`);
    current.set('sort', `${sort}`);
    current.set('sortBy', `${field}`);
    router.push(`/post?${current.toString()}`);
  };

  const handleRemove = async (id: string) => {
    try {
      if (!window.confirm('Are you sure you want to delete this Post?')) return;
      await postService.delete(id);
      message.success('Deleted Successfully.');
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    }
  };

  const handleUpdate = async (id: string, field: string, value: any) => {
    try {
      setLoading(true);
      await postService.update(id, { [field]: value });
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setLoading(false);
    }
  }

  const columns = [
    {
      title: 'Title',
      dataIndex: 'title',
      key: 'title',
      ellipsis: true,
      width: 200,
      sorter: true
    },
    {
      title: 'Slug',
      dataIndex: 'slug',
      key: 'slug',
      ellipsis: true,
      width: 200
    },
    {
      title: 'Status',
      dataIndex: 'status',
      key: 'status',
      render: (status: string, record: IPost) => {
        const { _id } = record;
        const isActive = status === 'active';
        const statusUpdate = isActive ? 'inactive' : 'active';
        return (<span style={{ cursor: 'pointer' }} onClick={() => handleUpdate(_id, 'status', statusUpdate)}><StatusBadge active={isActive} activeText="Active" inActiveText="Inactive" /></span>)
      },
      width: 100,
      sorter: true
    },
    {
      title: 'Updated At',
      dataIndex: 'updatedAt',
      key: 'updatedAt',
      render(date: Date) {
        return <span>{formatDate(date)}</span>;
      },
      width: 200,
      sorter: true
    },
    {
      title: 'Action',
      key: 'action',
      render: (record: any) => (
        <Space>
          <Link href={`/post/update/${record._id}`}>
            <Button type="primary">Update</Button>
          </Link>
          <Button onClick={() => handleRemove(record._id)}>Delete</Button>
        </Space>
      ),
      width: 200
    }
  ];

  return (
    <Table
      className="ant-border-space"
      columns={columns}
      dataSource={data}
      pagination={{
        pageSize: pageSize,
        total: total,
        current: page ? Number(page) : 1
      }}
      onChange={handleChange}
      scroll={{ x: 1200 }}
      loading={loading}
    />
  );
}
