'use client';

import {
  Form,
  Button,

  Input,
  message,
  Select
} from 'antd';
import { FormInstance } from 'antd/lib/form';
import { useEffect, useRef } from 'react';
import {
  useFormStatus,
  useFormState
} from 'react-dom';
import { redirect } from 'next/navigation';
import { ActionState, IPost } from 'src/interfaces';
import dynamic from 'next/dynamic';
import CategorySelection from '@components/category/category-selection';
import { STATUS } from 'src/constants';
import { slugValidation } from 'src/validations';
const SUNEDITOR = dynamic(() => import('@components/common/suneditor'), {
  ssr: false
});


interface IProps {
  data?: IPost | undefined;
  onFinish: any
}

export function FormPost({ data = undefined, onFinish }: IProps) {
  const [state, formAction] = useFormState<ActionState, IPost>(onFinish, {});
  const formRef = useRef<FormInstance>(null);
  const { pending } = useFormStatus();

  function setFormVal(field: string, val: any) {
    const instance = formRef.current as FormInstance;
    instance.setFieldsValue({
      [field]: val
    });
  }

  useEffect(() => {
    if (state.success) {
      message.success('Updated Successfully');
      redirect('/post');
    } else {
      state.message && message.error(state.message);
    }
  }, [state]);

  useEffect(() => {
    if (data) {
      formRef.current?.setFieldsValue(data);
    }
  }, [data]);

  return (
    <Form
      ref={formRef}
      onFinish={(values) => formAction({ ...data, ...values })}
      initialValues={{
        title: '',
        slug: '',
        content: '',
        seoTitle: '',
        seoKeywords: '',
        seoDesc: '',
        status: STATUS.ACTIVE
      }}
      layout="vertical"
    >
      <Form.Item
        name="title"
        label="Post Title"
        rules={[{ required: true, message: 'Enter the post title!' }]}
      >
        <Input placeholder="Title" />
      </Form.Item>
      <Form.Item
        name="slug"
        label="Slug"
        rules={[slugValidation]}
      >
        <Input placeholder="Slug" />
      </Form.Item>
      <Form.Item
        name="categoryIds"
        label="Categories"
      >
        <CategorySelection
          selectProps={{ mode: 'multiple', defaultValue: (data && data.categoryIds) || [] }}
          onChange={(field: string, value: any) => setFormVal(field, value)}
          fieldName='categoryIds'
          group="post"
        />
      </Form.Item>
      <Form.Item
        name="seoTitle"
        label="Title (SEO)"
      >
        <Input placeholder="Enter seo title" />
      </Form.Item>
      <Form.Item
        name="seoKeywords"
        label="Keyword (SEO)"
      >
        <Input placeholder="Enter keywords" />
      </Form.Item>
      <Form.Item
        name="seoDesc"
        label="Description (SEO)"
      >
        <Input placeholder="Enter SEO description" />
      </Form.Item>
      <Form.Item
        label="Content"
        name="content"
        rules={[{ required: true, message: 'Required!' }]}
      >
        <SUNEDITOR
          richTextEditorHtml={data?.content || ''}
          onRichTextEditorChange={({ html }: any) => setFormVal('content', html)}
          displayMode="EDIT"
        />
      </Form.Item>
      <Form.Item
        name="status"
        label="Status"
        rules={[{ required: true, message: 'Please select status!' }]}
      >
        <Select placeholder={'Select status'}>
          <Select.Option key="active" value="active">
            Active
          </Select.Option>
          <Select.Option key="inactive" value="inactive">
            Inactive
          </Select.Option>
        </Select>
      </Form.Item>
      <Form.Item>
        <Button
          type="primary"
          htmlType="submit"
          loading={pending}
          disabled={pending}
        >
          Submit
        </Button>
      </Form.Item>
    </Form>
  );
}
