'use client';

import {
  Form,
  Button,
  Select,
  Input,
  message
} from 'antd';
import { FormInstance } from 'antd/lib/form';
import { useEffect, useRef, useState } from 'react';
import {
  useFormStatus,
  useFormState
} from 'react-dom';
import { redirect } from 'next/navigation';
import { ActionState, IPerformer } from 'src/interfaces';
import ImageUpload from '@components/file/image-upload';
import { performerService } from '@services/performer.service';
import { getResponseError } from '@lib/utils';
import { STATUS } from 'src/constants';
import { slugValidation } from 'src/validations';

interface IProps {
  data?: IPerformer | undefined;
  onFinish: any
}

export function FormPerformer({ data = undefined, onFinish }: IProps) {
  const [uploading, setUploading] = useState(false);
  const [state, formAction] = useFormState<ActionState, IPerformer>(onFinish, {});
  const formRef = useRef<FormInstance>(null);
  const { pending } = useFormStatus();

  useEffect(() => {
    if (state.success) {
      message.success('Success');
      redirect('/pornstar');
    } else {
      state.message && message.error(state.message);
    }
  }, [state]);

  useEffect(() => {
    if (data) {
      formRef.current?.setFieldsValue(data);
    }
  }, [data]);

  const onFileSelected = async (field: string, file: File) => {
    try {
      setUploading(true);
      const { data: signUrlResp } = field === 'avatarId' ? await performerService.signAvatarUploadUrl() : await performerService.signCoverUploadUrl();
      if (signUrlResp.uploadUrl) {
        const resp = await performerService.uploadFile(signUrlResp.uploadUrl, file) as any;
        if (resp.status) {
          const { data: fileInfo } = await performerService.getFileUploadedInfo(signUrlResp.fileId);
          const formInstance = formRef.current as FormInstance;
          formInstance.setFieldsValue({
            [field]: fileInfo._id
          });
        }
      }
    } catch (error) {
      message.error(getResponseError(error));
    } finally {
      setUploading(false);
    }
  }

  return (
    <Form
      ref={formRef}
      onFinish={(values) => formAction({ ...data, ...values })}
      initialValues={{
        name: '',
        bio: '',
        gender: undefined,
        status: STATUS.ACTIVE,
        seoTitle: '',
        seoKeywords: '',
        seoDesc: ''
      }}
      layout="vertical"
    >
      <Form.Item
        name="name"
        label="Name"
        rules={[{ required: true, message: 'Please enter name!' }, { pattern: new RegExp(/^[a-zA-Z.+'-]+(?:\s[a-zA-Z.+'-]+)*\s?$/), message: 'Please enter a valid name' }]}
      >
        <Input />
      </Form.Item>
      <Form.Item
        name="username"
        label="Slug"
        rules={[slugValidation]}
      >
        <Input placeholder="Enter a custom URL" />
      </Form.Item>
      <Form.Item
        name="gender"
        label="Gender"
        rules={[{ required: true, message: 'Please select gender!' }]}
      >
        <Select placeholder="Gender">
          <Select.Option key="male" value="male">
            Male
          </Select.Option>
          <Select.Option key="female" value="female">
            Female
          </Select.Option>
          <Select.Option key="transgender" value="transgender">
            Transgender
          </Select.Option>
        </Select>
      </Form.Item>
      <Form.Item
        name="bio"
        label="Bio"
      >
        <Input.TextArea rows={5} />
      </Form.Item>
      <Form.Item
        name="status"
        label="Status"
        rules={[{ required: true, message: 'Please select status!' }]}
      >
        <Select placeholder="Status">
          <Select.Option key="active" value="active">
            Active
          </Select.Option>
          <Select.Option key="inactive" value="inactive">
            Inactive
          </Select.Option>
        </Select>
      </Form.Item>
      <Form.Item
        name="avatarId"
        label="Avatar"
        extra={<div>
          <p>Maximum file size up to {process.env.MAX_SIZE_IMAGE || 5}MB.</p>
          <p>Accept image/*</p>
          <p>Recommended image size 480x640 pixels</p>
        </div>
        }
      >
        <ImageUpload
          image={data?.avatarUrl}
          onFileSelected={(file: File) => onFileSelected('avatarId', file)}
        />
      </Form.Item>
      <Form.Item
        name="coverId"
        label="Cover"
        extra={<div>
          <p>Maximum file size up to {process.env.MAX_SIZE_IMAGE || 5}MB.</p>
          <p>Accept image/*</p>
          <p>Recommended image size 1600x400 pixels</p>
        </div>
        }
      >
        <ImageUpload
          image={data?.coverUrl}
          onFileSelected={(file: File) => onFileSelected('coverId', file)}
        />
      </Form.Item>
      <Form.Item
        name="seoTitle"
        label="Title (SEO)"
      >
        <Input placeholder="Enter seo title" />
      </Form.Item>
      <Form.Item
        name="seoKeywords"
        label="Keyword (SEO)"
      >
        <Input placeholder="Enter keywords" />
      </Form.Item>
      <Form.Item
        name="seoDesc"
        label="Description (SEO)"
      >
        <Input placeholder="Enter SEO description" />
      </Form.Item>
      <Form.Item>
        <Button
          type="primary"
          htmlType="submit"
          loading={pending || uploading}
          disabled={pending || uploading}
        >
          Submit
        </Button>
      </Form.Item>
    </Form>
  );
}
