'use client';

import { Button, Modal } from 'antd';
import { useCallback, useState } from 'react';
import {
  EyeOutlined
} from '@ant-design/icons';
import { VideoPlayer } from './video-player';

import s from './preview-modal.module.scss';

interface IProps {
  key: string;
  mediaType: string;
  url: string;
  poster?: string;
}

function isURL(str: string): boolean {
  try {
    new URL(str);
    return true;
  } catch {
    return false;
  }
}

function ensureHTTPS(url: string): string {
  if (url.startsWith('http://')) {
    return url.replace('http://', 'https://');
  }
  return url;
}

export default function MediaPreviewModal({ key, mediaType, url }: IProps) {
  const [isModalOpen, setIsModalOpen] = useState(false);

  const secureUrl = ensureHTTPS(url);

  const renderMedia = useCallback(() => {
    switch (mediaType) {
      case 'video':
        return (
          <VideoPlayer
            key={key}
            autoplay
            controls
            playsinline
            sources={[
              {
                src: secureUrl,
                type: 'video/mp4'
              }
            ]}
          />
        );
      case 'embeddedCode':
        if (isURL(secureUrl)) {
          return <iframe src={secureUrl} style={{ width: '100%', height: '100%' }} />;
        }
        return (
          <div
            className={s['embedded-wrapper']}
            dangerouslySetInnerHTML={{ __html: secureUrl }}
          ></div>
        );
      case 'iframe':
        return <iframe src={secureUrl} style={{ width: '100%', height: '100%' }} />;
      default:
        return null;
    }
  }, [secureUrl, mediaType]);

  return (
    <>
      <Button onClick={() => setIsModalOpen(true)}>
        <EyeOutlined /> Preview
      </Button>
      {isModalOpen && (
        <Modal
          centered
          open={isModalOpen}
          onCancel={() => setIsModalOpen(false)}
          footer={null}
          styles={{
            body: {
              paddingTop: 30,
              height: 400
            }
          }}
        >
          {renderMedia()}
        </Modal>
      )}
    </>
  );
}
