import { CameraOutlined, LoadingOutlined } from '@ant-design/icons';
import { message, Upload } from 'antd';
import { useState } from 'react';
import s from './image-upload.module.scss';

function getBase64(img: any, callback: Function) {
  const reader = new FileReader();
  reader.addEventListener('load', () => callback(reader.result));
  reader.readAsDataURL(img);
}

interface IProps {
  image?: string;
  uploadUrl?: string;
  headers?: any;
  onUploaded?: Function;
  options?: any;
  uploadImmediately?: boolean;
  onFileSelected?: Function;
}

export default function ImageUpload({
  image,
  uploadUrl,
  headers,
  onUploaded,
  options,
  uploadImmediately,
  onFileSelected = () => { }
}: IProps) {
  const [loading, setLoading] = useState(false);
  const [imageUrl, setImageUrl] = useState('');

  const beforeUpload = (file: File) => {
    const isMaxSize = file.size / 1024 / 1024 < (Number(process.env.MAX_SIZE_IMAGE || 5));
    if (!isMaxSize) {
      message.error(`Image must be smaller than ${process.env.MAX_SIZE_IMAGE || 5}MB!`);
      return false;
    }
    getBase64(file, (url: string) => {
      setImageUrl(url);
    });

    if (!uploadImmediately) {
      onFileSelected(file);
      return false;
    }
    return isMaxSize;
  };

  const handleChange = (info: any) => {
    if (info.file.status === 'uploading') {
      setLoading(true);
      return;
    }
    if (info.file.status === 'done') {
      // Get this url from response in real world.
      getBase64(info.file.originFileObj, (url: string) => {
        setImageUrl(url);
        setLoading(false);
        onUploaded
          && onUploaded({
            response: info.file.response,
            base64: imageUrl
          });
      });
    }
  };

  const uploadButton = (
    <div className='upload-btn'>
      {loading ? <LoadingOutlined /> : <CameraOutlined />}
    </div>
  );
  return (
    <Upload
      accept={'image/*'}
      name={options.fieldName || 'file'}
      listType="picture-card"
      disabled={loading}
      className={s['avatar-uploader']}
      showUploadList={false}
      action={uploadUrl}
      beforeUpload={beforeUpload}
      onChange={handleChange}
      headers={headers}
    >
      {(imageUrl || image) && <img src={imageUrl || image} alt="file" style={{ width: '100%' }} />}
      {uploadButton}
    </Upload>
  );
}

ImageUpload.defaultProps = {
  image: null,
  uploadUrl: null,
  headers: null,
  onUploaded: () => { },
  options: {},
  uploadImmediately: false,
  onFileSelected: () => { }
}