import parser from 'html-react-parser';
import dynamic from 'next/dynamic';
import {
  align,
  font,
  fontColor,
  fontSize,
  formatBlock,
  hiliteColor,
  horizontalRule,
  lineHeight,
  list,
  paragraphStyle,
  table,
  template,
  textStyle,
  image,
  link,
  video
} from 'suneditor/src/plugins';

import { postService } from '@services/post.service';
import { useState } from 'react';
import 'suneditor/dist/css/suneditor.min.css';
const SunEditor = dynamic(() => import('suneditor-react'), {
  ssr: false
});

interface IRichTextEditorProps {
  richTextEditorHtml: string;
  onRichTextEditorChange: Function;
  displayMode: any;
}

const baseOptions = {
  showPathLabel: false,
  minHeight: '50vh',
  maxHeight: '50vh',
  placeholder: 'Enter your content here!!!',
  plugins: [
    align,
    font,
    fontColor,
    fontSize,
    formatBlock,
    hiliteColor,
    horizontalRule,
    lineHeight,
    list,
    paragraphStyle,
    table,
    template,
    textStyle,
    image,
    link,
    video
  ],
  buttonList: [
    ['undo', 'redo'],
    ['font', 'fontSize', 'formatBlock'],
    ['paragraphStyle'],
    ['bold', 'underline', 'italic', 'strike', 'subscript', 'superscript'],
    ['fontColor', 'hiliteColor'],
    ['removeFormat'],
    '/', // Line break
    ['outdent', 'indent'],
    ['align', 'horizontalRule', 'list', 'lineHeight'],
    ['table', 'link', 'image'],
    ['video', 'fullScreen', 'showBlocks', 'codeView', 'print']
  ],
  font: [
    'Arial',
    'Calibri',
    'Comic Sans',
    'Courier',
    'Garamond',
    'Georgia',
    'Impact',
    'Lucida Console',
    'Palatino Linotype',
    'Segoe UI',
    'Tahoma',
    'Times New Roman',
    'Trebuchet MS',
    'Helvetica'
  ]
}

export default function SunEditorHtml({ richTextEditorHtml, onRichTextEditorChange, displayMode }: IRichTextEditorProps) {
  const [richTextEditor, setRichTextEditor] = useState(localStorage.getItem('richText') || richTextEditorHtml || '');
  const [options] = useState(baseOptions);

  const handleChange = (content: any) => {
    setRichTextEditor(content);
    onRichTextEditorChange({
      html: `${content}`
    })
  };

  const uploadPhoto = (files: File[], info: any, uploadHandler: any) => {
    postService.signImageUploadUrl().then(async (resp) => {
      const { data } = resp;
      if (data && data.uploadUrl) {
        const uploadResp = await postService.uploadFile(data.uploadUrl, files[0]) as any;
        if (uploadResp.status) {
          const fileInfo = await postService.getFileUploadedInfo(data.fileId) as any;
          const { data: file } = fileInfo;

          if (file.url) {
            const response = {
              result: [
                {
                  url: file.url,
                  type: file.mediaType
                }
              ]
            };

            uploadHandler(response);
          }
        }
      }
    });
    return false;
  }

  return (
    <div>
      {displayMode === 'EDIT' ? (
        <SunEditor
          autoFocus={false}
          lang="en"
          // eslint-disable-next-line react/destructuring-assignment
          setOptions={options}
          defaultValue={richTextEditorHtml}
          onChange={handleChange}
          // eslint-disable-next-line react/jsx-no-bind
          onImageUploadBefore={uploadPhoto}
        />
      ) : richTextEditor && (
        <div>
          <div className="sun-editor">
            <div className="sun-editor-editable">{parser(richTextEditor)}</div>
          </div>
        </div>
      )}
    </div>
  );
}