'use client';

import { ICategory } from 'src/interfaces';
import { getResponseError } from 'src/lib';
import { categoryService } from 'src/services';
import { Button, Space, Table, TableProps, message } from 'antd';
import Link from 'next/link';
import { useRouter, useSearchParams } from 'next/navigation';
import StatusBadge from '@components/common/status-badge';
import { SorterResult } from 'antd/es/table/interface';
import { formatDate } from '@lib/date';
import { useEffect, useState } from 'react';
import { TablePaginationConfig } from 'antd/lib';

type Props = {
  data: ICategory[];
  total: number;
  pageSize: number;
};

export function CategoryTableList({ data, total, pageSize }: Props) {
  const [selectedRowKeys, setSelectedRowKeys] = useState<React.Key[]>([]);
  const [loading, setLoading] = useState(false);
  const [updating, setUpdating] = useState(false);
  const [categories, setCategories] = useState([] as ICategory[]);
  const router = useRouter();

  const searchParams = useSearchParams();
  const page = searchParams.get('page');

  useEffect(() => {
    setCategories(data);
  }, [data]);

  const handleChange: TableProps<ICategory>['onChange'] = (pagination: TablePaginationConfig, filters, sorter) => {
    const { field = '', order } = sorter as SorterResult<ICategory>;
    const sort = order ? (order === 'descend' ? 'desc' : 'asc') : '';
    const current = new URLSearchParams(Array.from(searchParams.entries()));
    current.set('page', `${pagination.current || 1}`);
    if (sort && field) {
      current.set('sort', `${sort}`);
      current.set('sortBy', `${field}`);
    }
    router.push(`/category?${current.toString()}`);
  };

  const handleRemove = async (id: string) => {
    try {
      if (!window.confirm('Are you sure you want to delete this Category?')) return;
      setLoading(true);
      await categoryService.delete(id);
      const newData = categories.filter(item => item._id !== id);
      message.success('Deleted Successfully.');
      if (newData.length) {
        setCategories(newData);
      } else {
        router.refresh();
      }
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setLoading(false);
    }
  };

  const removeALl = async () => {
    // ajax request after empty completing
    try {
      if (!window.confirm('Are you sure you want to delete all selected Categories?')) return;
      setLoading(true);
      await selectedRowKeys.reduce(async (lp, id) => {
        await lp;
        await categoryService.delete(id as string);
        return Promise.resolve();
      }, Promise.resolve());
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setLoading(false);
      setSelectedRowKeys([]);
    }
  };

  const onSelectChange = (newSelectedRowKeys: React.Key[]) => {
    setSelectedRowKeys(newSelectedRowKeys);
  };

  const handleUpdate = async (id: string, field: string, value: any) => {
    try {
      setUpdating(true);
      await categoryService.update(id, { [field]: value });
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setUpdating(false)
    }
  }

  const rowSelection = {
    selectedRowKeys,
    onChange: onSelectChange
  };
  const hasSelected = selectedRowKeys.length > 0;

  const dataSource = categories.map(d => ({
    ...d,
    key: d._id
  })) as ICategory[];

  const columns = [
    {
      title: 'Title',
      dataIndex: 'title',
      key: 'title',
      sorter: true,
      ellipsis: true
    },
    {
      title: 'Group',
      dataIndex: 'group',
      key: 'group',
      sorter: true
    },
    // {
    //   title: 'Description',
    //   dataIndex: 'description',
    //   key: 'description',
    //   sorter: true
    // },
    {
      title: 'Status',
      dataIndex: 'status',
      key: 'status',
      // sorter: true,
      render: (status: string, record: ICategory) => {
        const { _id } = record;
        const isActive = status === 'active';
        const statusUpdate = isActive ? 'inactive' : 'active';
        return (<span style={{ cursor: 'pointer' }} onClick={() => handleUpdate(_id, 'status', statusUpdate)}><StatusBadge active={isActive} activeText="Active" inActiveText="Inactive" /></span>)
      }
    },
    {
      title: 'Ordering',
      dataIndex: 'ordering',
      key: 'ordering',
      sorter: true
    },
    {
      title: 'Updated At',
      dataIndex: 'updatedAt',
      key: 'updatedAt',
      render(date: Date) {
        return <span>{formatDate(date)}</span>;
      }
    },
    {
      title: 'Action',
      key: 'action',
      render: (record: any) => (
        <Space>
          <Link href={`/category/update/${record._id}`}>
            <Button type="primary">Update</Button>
          </Link>
          <Button onClick={() => handleRemove(record._id)}>Delete</Button>
        </Space>
      ),
      width: 200
    }
  ];

  return (
    <div>
      <div style={{ marginBottom: 16 }}>
        <Button type="primary" danger onClick={removeALl} disabled={!hasSelected} loading={loading}>
          Delete Categories
        </Button>
        <span style={{ marginLeft: 8 }}>
          {hasSelected ? `Selected ${selectedRowKeys.length} items` : ''}
        </span>
      </div>
      <Table
        className="ant-border-space"
        columns={columns}
        dataSource={dataSource}
        pagination={{
          pageSize: pageSize,
          total: total,
          current: page ? Number(page) : 1
        }}
        onChange={handleChange}
        scroll={{ x: 1200 }}
        rowSelection={rowSelection}
        loading={loading || updating}
      />
    </div>

  );
}
