'use client';

import { Form, Button, Input, message, Select, InputNumber } from 'antd';
import { FormInstance } from 'antd/lib/form';
import { useEffect, useRef, useState } from 'react';
import { useFormState } from 'react-dom';
import { redirect, useSearchParams } from 'next/navigation';
import { ActionState, ICategory } from 'src/interfaces';
import ImageUpload from '@components/file/image-upload';
import { getResponseError } from '@lib/utils';
import { CATEGORY_GROUP, STATUS } from 'src/constants';
import { slugValidation } from 'src/validations';
import { channelService } from '@services/channel.service';

interface IProps {
  data?: ICategory | undefined;
  onFinish: any;
}

export function FormCategory({ data = undefined, onFinish }: IProps) {
  const [uploading, setUploading] = useState(false);
  const [submitting, setSubmitting] = useState(false);
  const params = useSearchParams();
  const [state, formAction] = useFormState<ActionState, ICategory>(
    onFinish,
    {}
  );
  const formRef = useRef<FormInstance>(null);

  useEffect(() => {
    if (state.success || state.error || state.message) {
      setSubmitting(false);
    }
    if (state.success) {
      message.success('Updated Successfully');
      if (data) {
        redirect(`/category?group=${data.group || 'video'}`);
      } else {
        redirect(`/category?group=${params.get('group') || 'video'}`);
      }
    } else {
      state.message && message.error(state.message);
    }
  }, [state, data, params]);

  useEffect(() => {
    if (data) {
      formRef.current?.setFieldsValue(data);
    }
  }, [data]);

  const onFileSelected = async (field: string, file: File) => {
    try {
      setUploading(true);
      const { data: respData } = await channelService.signImageUploadUrl();
      if (respData.uploadUrl) {
        const resp = (await channelService.uploadFile(
          respData.uploadUrl,
          file
        )) as any;
        if (resp.status) {
          const { data: fileInfo } = await channelService.getFileUploadedInfo(
            respData.fileId
          );
          const formInstance = formRef.current as FormInstance;
          formInstance.setFieldsValue({
            [field]: fileInfo._id
          });
        }
      }
    } catch (error) {
      message.error(getResponseError(error));
    } finally {
      setUploading(false);
    }
  };

  const group = data && data.group ? data.group : (params.get('group') || 'video')

  return (
    <Form
      ref={formRef}
      onFinish={(values) => {
        setSubmitting(true);
        formAction({ ...data, ...values });
      }}
      initialValues={
        {
          title: '',
          description: '',
          slug: '',
          seoTitle: '',
          seoKeywords: '',
          seoDesc: '',
          status: STATUS.ACTIVE,
          ordering: 0,
          group: params.get('group') || 'video'
        } as ICategory
      }
      layout="vertical"
    >
      <Form.Item
        name="title"
        label="Category Name"
        rules={[{ required: true, message: 'Please enter a category Name.' }]}
      >
        <Input placeholder="Title" />
      </Form.Item>
      <Form.Item
        name="slug"
        label="Slug"
        help="Spaces and special characters are not allowed. Please replace them with hyphens or underscores."
        rules={[slugValidation]}
      >
        <Input placeholder="Enter a Custom URL" />
      </Form.Item>
      <Form.Item name="description" label="Description">
        <Input.TextArea rows={5} placeholder="Description" />
      </Form.Item>
      <Form.Item hidden={group === 'post'} name="seoTitle" label="Title (SEO)">
        <Input placeholder="Enter seo title" />
      </Form.Item>
      <Form.Item hidden={group === 'post'} name="seoKeywords" label="Keyword (SEO)">
        <Input placeholder="Enter keywords" />
      </Form.Item>
      <Form.Item hidden={group === 'post'} name="seoDesc" label="Description (SEO)">
        <Input placeholder="Enter SEO description" />
      </Form.Item>
      <Form.Item
        name="ordering"
        label="Ordering"
        rules={[
          { required: true, message: 'Required!' },
          {
            type: 'number',
            min: 0,
            message: 'Ordering is not valid',
            transform(value) {
              return Number(value);
            }
          }
        ]}
      >
        <InputNumber min={0} style={{ width: '100%' }} placeholder="Enter ordering" />
      </Form.Item>
      <Form.Item
        name="status"
        label="Status"
        rules={[{ required: true, message: 'Please select status!' }]}
      >
        <Select placeholder="Status">
          <Select.Option key="active" value="active">
            Active
          </Select.Option>
          <Select.Option key="inactive" value="inactive">
            Inactive
          </Select.Option>
        </Select>
      </Form.Item>
      <Form.Item
        name="group"
        label="Group"
        hidden
        rules={[{ required: true, message: 'Please select group!' }]}
      >
        <Select placeholder="Group" disabled>
          {CATEGORY_GROUP.map((g) => (
            <Select.Option key={g.value} value={g.value}>
              {g.label}
            </Select.Option>
          ))}
        </Select>
      </Form.Item>
      <Form.Item dependencies={['group']} >
        {(form) =>
          form.getFieldValue('group') === 'video' && (
            <Form.Item extra={<div>
              <p>Maximum file size up to {process.env.MAX_SIZE_IMAGE || 5}MB.</p>
              <p>Accept image/*</p>
              <p>Recommended image size 480x640 pixels</p>
            </div>
            } name="posterId" label="Category Image">
              <ImageUpload
                image={data?.posterUrl}
                onFileSelected={(file: File) =>
                  onFileSelected('posterId', file)
                }
              />
            </Form.Item>
          )
        }
      </Form.Item>

      <Form.Item>
        <Button
          type="primary"
          htmlType="submit"
          loading={submitting || uploading}
          disabled={submitting || uploading}
        >
          Submit
        </Button>
      </Form.Item>
    </Form>
  );
}
