'use client';

import { IAd } from 'src/interfaces';
import { getResponseError } from 'src/lib';
import { adService } from 'src/services';
import { Button, Space, Table, TablePaginationConfig, message } from 'antd';
import Link from 'next/link';
import { useRouter, useSearchParams } from 'next/navigation';
import StatusBadge from '@components/common/status-badge';
import { formatDate } from '@lib/date';
import { AD_POSITION } from 'src/constants';
import { useState } from 'react';
import { TableProps } from 'antd/lib';
import { SorterResult } from 'antd/es/table/interface';

type Props = {
  data: IAd[];
  total: number;
  pageSize: number;
};

export function AdTableList({ data, total, pageSize }: Props) {
  const [selectedRowKeys, setSelectedRowKeys] = useState<React.Key[]>([]);
  const [loading, setLoading] = useState(false);
  const [updating, setUpdating] = useState(false);
  const router = useRouter();

  const searchParams = useSearchParams();
  const page = searchParams.get('page');

  const handleChange: TableProps<IAd>['onChange'] = (pagination: TablePaginationConfig, filters, sorter) => {
    const { field = '', order } = sorter as SorterResult<IAd>;
    const sort = order ? (order === 'descend' ? 'desc' : 'asc') : '';
    const current = new URLSearchParams(Array.from(searchParams.entries()));
    current.set('page', `${pagination.current || 1}`);
    current.set('sort', `${sort}`);
    current.set('sortBy', `${field}`);
    router.push(`/ad?${current.toString()}`);
  };

  const handleRemove = async (id: string) => {
    try {
      if (!window.confirm('Are you sure you want to delete this Ad?')) return;
      await adService.delete(id);
      message.success('Deleted Successfully.');
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    }
  };

  const removeALl = async () => {
    // ajax request after empty completing
    try {
      if (!window.confirm('Are you sure you want to delete all selected Ads?')) return;
      setLoading(true);
      await selectedRowKeys.reduce(async (lp, id) => {
        await lp;
        await adService.delete(id as string);
        return Promise.resolve();
      }, Promise.resolve());
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setLoading(false);
      setSelectedRowKeys([]);
    }
  };

  const onSelectChange = (newSelectedRowKeys: React.Key[]) => {
    setSelectedRowKeys(newSelectedRowKeys);
  };

  const handleUpdate = async (id: string, field: string, value: any) => {
    try {
      setUpdating(true);
      await adService.update(id, { [field]: value });
      router.refresh();
    } catch (e) {
      const error = await Promise.resolve(e);
      message.error(getResponseError(error));
    } finally {
      setUpdating(false)
    }
  }

  const rowSelection = {
    selectedRowKeys,
    onChange: onSelectChange
  };
  const hasSelected = selectedRowKeys.length > 0;

  const dataSource = data?.map(d => ({
    ...d,
    key: d._id
  }))

  const columns = [
    {
      title: 'Thumbnail',
      key: 'thumbnailUrl',
      render: (record: IAd) => {
        const { thumbnailUrl, scriptCode } = record;
        if (scriptCode) {
          return <div dangerouslySetInnerHTML={{ __html: scriptCode }} />
        }

        return (
          <Space>
            {thumbnailUrl && <img src={thumbnailUrl} alt="ad thumbnail" />}
          </Space>
        )
      },
      // fixed: 'left' as FixedType,
      width: 225
    },
    {
      title: 'Name',
      dataIndex: 'name',
      key: 'name',
      ellipsis: true,
      sorter: true,
      width: 225
    },
    // {
    //   title: 'Type',
    //   dataIndex: 'type',
    //   key: 'type',
    //   sorter: true,
    //   width: 100
    // },
    {
      title: 'Position',
      dataIndex: 'position',
      key: 'position',
      render(position: string) {
        return <span>{AD_POSITION[position]}</span>;
      },
      sorter: true,
      ellipsis: true,
      width: 200
    },
    // {
    //   title: 'Ad Url',
    //   dataIndex: 'adUrl',
    //   key: 'adUrl',
    //   ellipsis: true
    // },
    {
      title: 'Description',
      dataIndex: 'description',
      key: 'description',
      ellipsis: true,
      width: 250
    },
    {
      title: 'Updated At',
      dataIndex: 'updatedAt',
      key: 'updatedAt',
      render(date: Date) {
        return <span>{formatDate(date)}</span>;
      },
      width: 200,
      sorter: true
    },
    {
      title: 'Status',
      dataIndex: 'status',
      key: 'status',
      render: (status: string, record: IAd) => {
        const { _id } = record;
        const isActive = status === 'active';
        const statusUpdate = isActive ? 'inactive' : 'active';
        return (<span style={{ cursor: 'pointer' }} onClick={() => handleUpdate(_id, 'status', statusUpdate)}><StatusBadge active={isActive} activeText="Active" inActiveText="Inactive" /></span>)
      },
      sorter: true,
      width: 100
    },
    {
      title: 'Action',
      key: 'action',
      render: (record: any) => (
        <Space>
          <Link href={`/ad/update/${record._id}`}>
            <Button type="primary">Update</Button>
          </Link>
          <Button onClick={() => handleRemove(record._id)}>Delete</Button>
        </Space>
      ),
      width: 200
    }
  ];

  return (
    <div>
      <div style={{ marginBottom: 16 }}>
        <Button type="dashed" danger onClick={removeALl} disabled={!hasSelected} loading={loading}>
          Delete Ads
        </Button>
        <span style={{ marginLeft: 8 }}>
          {hasSelected ? `Selected ${selectedRowKeys.length} items` : ''}
        </span>
      </div>
      <Table
        className="ant-border-space"
        columns={columns}
        dataSource={dataSource}
        pagination={{
          pageSize: pageSize,
          total: total,
          current: page ? Number(page) : 1
        }}
        onChange={handleChange}
        scroll={{ x: 1500 }}
        rowSelection={rowSelection}
        loading={loading || updating}
      />
    </div>
  );
}
