'use client';

import {
  Form,
  Button,
  Select,
  Input,
  message,
  Divider
} from 'antd';
import { FormInstance } from 'antd/lib/form';
import { useEffect, useRef, useState } from 'react';
import {
  useFormState
} from 'react-dom';
import { redirect } from 'next/navigation';
import { ActionState, IAd } from 'src/interfaces';
import ImageUpload from '@components/file/image-upload';
import { getResponseError } from '@lib/utils';
import { AD_POSITION_OPTIONS, AD_TYPE, AD_TYPE_OPTIONS, STATUS } from 'src/constants';
import MediaPreviewModal from '@components/media/preview-modal';
import FileUpload from '@components/file/file-upload';
import { adService } from '@services/ad.service';

interface IProps {
  data?: IAd | undefined;
  onFinish: any
}

export function FormAd({ data = undefined, onFinish }: IProps) {
  const [uploading, setUploading] = useState(false);
  const [isAdStandard, setIsAdStandard] = useState(false);
  const [isAdScriptCode, setIsAdScriptCode] = useState(false);
  const [isAdImage, setIsAdImage] = useState(false);
  const [isAdVideo, setIsAdVideo] = useState(false);
  const [state, formAction] = useFormState<ActionState, IAd>(onFinish, {});
  const formRef = useRef<FormInstance>(null);
  const [form] = Form.useForm();
  const type = Form.useWatch('type', form);
  const [submitting, setSubmitting] = useState(false);

  useEffect(() => {
    if (state.success || state.error) {
      setSubmitting(false);
    }
    if (state.success) {
      message.success('Updated Successfully');
      redirect('/ad');
    } else {
      state.message && message.error(state.message);
    }
  }, [state]);

  useEffect(() => {
    if (data) {
      formRef.current?.setFieldsValue(data);
      setIsAdStandard([AD_TYPE.SCRIPT_CODE, AD_TYPE.IMAGE].includes(data.type));
    }
  }, [data]);

  const onFileSelected = async (field: string, mediaType: string, file: File) => {
    try {
      setUploading(true);
      const { data: signUrlResp } = await adService.signFileUploadUrl({ mediaType });
      if (signUrlResp.uploadUrl) {
        const resp = await adService.uploadFile(signUrlResp.uploadUrl, file) as any;
        if (resp.status) {
          const { data: fileInfo } = await adService.getFileUploadedInfo(signUrlResp.fileId);
          const formInstance = formRef.current as FormInstance;
          formInstance.setFieldsValue({
            [field]: fileInfo._id
          });
        }
      }
    } catch (error) {
      message.error(getResponseError(error));
    } finally {
      setUploading(false);
    }
  }

  useEffect(() => {
    setIsAdStandard([AD_TYPE.SCRIPT_CODE, AD_TYPE.IMAGE].includes(type));
    setIsAdImage([AD_TYPE.IMAGE].includes(type));
    setIsAdScriptCode([AD_TYPE.SCRIPT_CODE].includes(type));
    setIsAdVideo([AD_TYPE.VIDEO].includes(type));
  }, [type])

  return (
    <Form
      ref={formRef}
      onFinish={(values) => {
        setSubmitting(true);
        formAction({ ...data, ...values })
      }}
      initialValues={{
        name: '',
        description: '',
        status: STATUS.ACTIVE,
        type: AD_TYPE.IMAGE
      }}
      layout="vertical"
      form={form}
    >
      <Form.Item
        name="type"
        label="Select ad type"
        rules={[{ required: true, message: 'Required!' }]}
      >
        <Select
          options={AD_TYPE_OPTIONS}
          placeholder="Select ad type"
        />
      </Form.Item>
      <Form.Item
        name="name"
        label="Name"
        rules={[{ required: true, message: 'Required!' }]}
      >
        <Input placeholder="Name" />
      </Form.Item>
      <Form.Item
        name="description"
        label="Description"
      >
        <Input.TextArea rows={5} placeholder="Description" />
      </Form.Item>
      {!isAdScriptCode && (
        <Form.Item
          name="adUrl"
          label="Ad URL"
          rules={[{ pattern: new RegExp(/(http|https):\/\/(\w+:{0,1}\w*)?(\S+)(:[0-9]+)?(\/|\/([\w#!:.?+=&%!\-\/]))?/), message: 'Url invalid' }]}
        >
          <Input placeholder="Ad URL" />
        </Form.Item>
      )}
      {isAdStandard &&
        (<>
          <Form.Item
            name="position"
            label="Select ad position"
            rules={[{ required: true, message: 'Required!' }]}
          >
            <Select
              options={AD_POSITION_OPTIONS}
              placeholder="Select ad position"
            />
          </Form.Item>
          {isAdImage &&
            <Form.Item
              name="fileId"
              label="Ad image"
              help={
              <div>
                <p>Recommended Banner Size</p>
                <ul style={{ padding: 20}}>
                  <li>Maximum file size up to 5MB</li>
                  <li>Accept image/*</li>
                  <li>Home: 1600 x 400 pixels</li>
                  <li>Footer: 300 x 300 pixels</li>
                  <li>Video Detail Page: 300 x 300 pixels</li>
                  <li>Under Trending: 1500 x 300 pixels </li>
                  <li>Under New Video: 1500 x 300 pixels</li>
                  <li>Under Video Player: 1500 x 300 pixels</li>
                </ul>
              </div>

              }
              rules={[{ required: true, message: 'Please select file!' }]}
            >
              <ImageUpload
                image={data?.fileUrl}
                onFileSelected={(file: File) => onFileSelected('fileId', 'image', file)}
              />
            </Form.Item>
          }
          {isAdScriptCode &&
            <Form.Item
              name="scriptCode"
              label="Script code"
              rules={[{ required: true, message: 'Required!' }]}
            >
              <Input.TextArea rows={5} placeholder="Script code" />
            </Form.Item>
          }
        </>)
      }
      {isAdVideo &&
        <Form.Item
          name="fileId"
          label="Upload video"
          rules={[{ required: true, message: 'Please select file!' }]}
        >
          {data && data.fileUrl &&
            <>
              <MediaPreviewModal key={data._id} mediaType="video" url={data.fileUrl} />
              <Divider />
            </>
          }
          <FileUpload maxSize={1024} onFileSelected={(file: File) => onFileSelected('fileId', 'video', file)} accept="video/*" uploadImmediately={false} />
        </Form.Item>
      }
      <Form.Item
        name="status"
        label="Status"
        rules={[{ required: true, message: 'Please select status!' }]}
      >
        <Select placeholder="Status">
          <Select.Option key="active" value="active">
            Active
          </Select.Option>
          <Select.Option key="inactive" value="inactive">
            Inactive
          </Select.Option>
        </Select>
      </Form.Item>

      <Form.Item>
        <Button
          type="primary"
          htmlType="submit"
          loading={submitting || uploading}
          disabled={submitting || uploading}
        >
          Submit
        </Button>
      </Form.Item>
    </Form>
  );
}
