import Page from 'src/components/layout/page';
import { categoryService } from 'src/services';
import { ISearch } from 'src/interfaces';
import { cookies } from 'next/headers';
import SearchFilter from 'src/components/common/search-filter';
import Link from 'next/link';
import { Button } from 'antd';
import { CategoryTableList } from '@components/category/table-list';
import { capitalizeFirstLetter } from '@lib/string';
import { redirect } from 'next/navigation';

interface SearchParams extends ISearch {
  q: string;
  group: string;
}

interface IProps {
  searchParams: SearchParams;
}

export const metadata = {
  title: 'Categories'
};

export default async function CategoryPage({
  searchParams: { page = 1, q = '', sort = 'asc', sortBy = 'ordering', pageSize = 10, group = 'video' }
}: IProps) {
  const cookieStore = cookies();
  const token = cookieStore.get('token');

  const { error, data } = await categoryService.search(
    { limit: pageSize, offset: (page - 1) * pageSize, q, sort, sortBy, group },
    {
      Authorization: token?.value || ''
    }
  );

  if (error) return null;
  const { total } = data as any;
  const totalPages = Math.ceil(total / pageSize);

  if (page > totalPages && total > 0) {
    redirect(`/category?group=${group}&page=${totalPages}`);
  }

  return (
    <Page
      title={`Manage ${capitalizeFirstLetter(group)} Category`}
      extra={
        <Link href={{ pathname: '/category/create', query: { group } }}>
          <Button type="primary">Add a new Category</Button>
        </Link>
      }
      breadcrumbs={[
        {
          title: 'Dashboard',
          key: 'dashboard',
          href: '/'
        },
        {
          title: `Manage ${capitalizeFirstLetter(group)} Category`,
          key: 'category'
        }
      ]}
    >
      <SearchFilter />
      <CategoryTableList {...(data as any)} pageSize={pageSize} />
    </Page>
  );
}
