import Page from 'src/components/layout/page';
import { adService } from 'src/services';
import { cookies } from 'next/headers';
import SearchFilter from 'src/components/common/search-filter';
import Link from 'next/link';
import { Button } from 'antd';
import { AdTableList } from '@components/ad/table-list';
import { notFound, redirect } from 'next/navigation';

type SearchParamsType = {
  page?: string;
  q?: string;
  sort?: string;
  sortBy?: string;
  status?: string;
  gender?: string;
};

// Define page props according to Next.js 15 standards
interface Props {
  params: Record<string, string>;
  searchParams: SearchParamsType;
};

export const metadata = {
  title: 'Ads'
};

export default async function AdPage({
  searchParams
}: Props) {
  const page = Number(searchParams.page) || 1;
  const q = searchParams.q || '';
  const sort = searchParams.sort || '';
  const sortBy = searchParams.sortBy || '';
  const cookieStore = await cookies();
  const token = cookieStore.get('token');

  const pageSize = 10;

  const { error, data } = await adService.search(
    { limit: pageSize, offset: (page - 1) * pageSize, q, sort, sortBy },
    {
      Authorization: token?.value || ''
    }
  );

  if (error) return notFound();
  const { total } = data as any || 0;
  const totalPages = Math.ceil(total / pageSize);

  if (page > totalPages && total > 0) {
    redirect(`/ad?page=${totalPages}`);
  }

  return (
    <Page
      title="Manage Ad"
      extra={
        <Link href="/ad/create">
          <Button type="primary">Add a new Ad</Button>
        </Link>
      }
      breadcrumbs={[
        {
          title: 'Dashboard',
          key: 'dashboard',
          href: '/'
        },
        {
          title: 'Manage Ad',
          key: 'ad'
        }
      ]}
    >
      <SearchFilter />
      <AdTableList {...(data as any)} pageSize={pageSize} />
    </Page>
  );
}
