
import { Inject, Injectable, forwardRef } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { compact, merge } from 'lodash';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import {
  EVENT, VIDEO_CHANNEL, VIDEO_FILE_CHANNEL
} from 'src/constants';
import {
  EntityNotFoundException, ForbiddenException, QueueMessageService, createAlias, isObjectId, randomString, replaceSpace, toObjectId
} from 'src/core';
import { ProfileDto, VideoDto } from 'src/dtos';
import { SignedUploadRequestPayload } from 'src/file-module/payloads';
import { FileService } from 'src/file-module/services';
import {
  VideoCreatePayload, VideoUpdatePayload
} from 'src/payloads';
import { Video, VideoDocument } from 'src/schemas';

import { CategorySearchService } from '../category';
import { PerformerSearchService } from '../performer';
import { ReactionService } from '../reaction';
import { TagService } from '../tag';

@Injectable()
export class VideoService {
  constructor(
    @InjectModel(Video.name) private readonly VideoModel: Model<VideoDocument>,
    private readonly fileService: FileService,
    private readonly queueMessageService: QueueMessageService,
    @Inject(forwardRef(() => ReactionService))
    private readonly reactionService: ReactionService,
    private readonly categorySearchService: CategorySearchService,
    private readonly performerSearchService: PerformerSearchService,
    private readonly tagService: TagService
  ) {}

  public async findById(id: string | ObjectId, groups = []): Promise<VideoDto> {
    const item = await this.VideoModel.findById(id);
    if (!item) return null;

    return plainToInstance(VideoDto, item.toObject(), { groups });
  }

  public async create(
    payload: VideoCreatePayload,
    currentProfile?: ProfileDto
  ): Promise<VideoDto> {
    let slug = replaceSpace(payload.slug || createAlias(payload.title));
    const exists = await this.VideoModel.findOne({ slug });
    if (exists) {
      slug = `${slug}-${randomString(5)}`;
    }
    // filter created tags to get ID
    const tags = [];
    let tagIds = [];
    if (payload.tagIds?.length) {
      await this.tagService.createFromList(payload.tagIds);
      const tagsDB = await this.tagService.findFromList(payload.tagIds);
      if (tagsDB.length) {
        tagIds = tagsDB.map((t) => t._id);
        tags.push(...tagsDB.map((t) => t.name));
      }
    }
    const video = await this.VideoModel.create({
      ...payload,
      tags,
      tagIds,
      slug,
      createdBy: currentProfile?._id,
      status: payload.isByMember ? 'inactive' : 'active',
      verified: true
    });
    await this.queueMessageService.publish(VIDEO_CHANNEL, {
      eventName: EVENT.CREATED,
      data: plainToInstance(VideoDto, video.toObject())
    });
    return plainToInstance(VideoDto, video.toObject());
  }

  public async findByIdOrSlug(id: string | ObjectId): Promise<VideoDocument> {
    const query = id instanceof ObjectId || isObjectId(id) ? { _id: id } : { slug: id };
    return this.VideoModel.findOne(query);
  }

  public async update(
    id: string | ObjectId,
    payload: VideoUpdatePayload,
    currentProfile: ProfileDto
  ): Promise<VideoDto> {
    const video = await this.VideoModel.findById(id);
    if (!video) throw new EntityNotFoundException();

    if (
      !currentProfile.isAdmin()
      && !video.createdBy.equals(currentProfile._id)
    ) throw new ForbiddenException();

    let slug = replaceSpace(payload.slug || createAlias(payload.title));
    const exists = await this.VideoModel.findOne({
      slug,
      _id: {
        $ne: video._id
      }
    });
    if (exists) {
      slug = `${slug}-${randomString(5)}`;
    }
    const oldFileId = video.fileId;
    const { tagIds: tagsPayload, ...rests } = payload;
    merge(video, rests);
    video.slug = slug;
    if (typeof payload.performerIds !== 'undefined') {
      video.performerIds = payload.performerIds;
    }
    // filter created tags to get ID
    const tags = [];
    let tagIds = [];
    if (tagsPayload?.length) {
      await this.tagService.createFromList(tagsPayload);
      const tagsDB = await this.tagService.findFromList(tagsPayload);
      if (tagsDB.length) {
        tagIds = tagsDB.map((t) => t._id);
        tags.push(...tagsDB.map((t) => t.name));
      }
    }
    // if (typeof payload.tags !== 'undefined') {
    //   video.tags = payload.tags;
    // }
    video.tags = tags;
    video.tagIds = tagIds;
    video.markModified('tagIds');
    video.markModified('tags');
    if (typeof payload.categoryIds !== 'undefined') {
      video.categoryIds = payload.categoryIds;
    }
    video.updatedAt = new Date();
    await video.save();
    if (payload.fileId && `${payload.fileId}` === `${oldFileId}`) {
      await this.fileService.deleteFile(oldFileId);
    }
    await this.queueMessageService.publish(VIDEO_CHANNEL, {
      eventName: EVENT.UPDATED,
      data: plainToInstance(VideoDto, video.toObject())
    });
    return plainToInstance(VideoDto, video.toObject());
  }

  public async delete(
    id: string | ObjectId,
    currentProfile: ProfileDto
  ): Promise<any> {
    const video = await this.VideoModel.findById(id);
    if (!video) throw new EntityNotFoundException();

    if (
      !currentProfile.isAdmin()
      && !video.createdBy.equals(currentProfile._id)
    ) throw new ForbiddenException();

    await video.deleteOne();

    if (video.fileId) {
      await this.fileService.deleteFile(video.fileId);
    }
    await this.queueMessageService.publish(VIDEO_CHANNEL, {
      eventName: EVENT.DELETED,
      data: plainToInstance(VideoDto, video.toObject())
    });
    this.queueMessageService.publish('REACTION_VIDEO', {
      data: video,
      eventName: EVENT.DELETED
    });
    return true;
  }

  public async getDetails(
    id: string | ObjectId,
    user?: ProfileDto
  ): Promise<VideoDto> {
    const video = await this.findByIdOrSlug(id);
    if (!video) throw new EntityNotFoundException();

    const dto = plainToInstance(VideoDto, video.toObject());
    if (user && user._id) {
      const info = await this.reactionService.getReactionInfo(dto._id, user);
      dto.setReactionInfo(info);
    }
    const { performerIds, categoryIds, tagIds } = video;
    const [performers, categories, tags] = await Promise.all([
      performerIds && performerIds.length > 0
        ? this.performerSearchService.findByIds(performerIds, user)
        : [],
      categoryIds && categoryIds.length > 0
        ? this.categorySearchService.findByIds(categoryIds)
        : [],
      tagIds && tagIds.length > 0 ? this.tagService.findByIds(tagIds) : []
    ]);

    if (performers.length) {
      const performerStringIds = compact(video.performerIds).map((pId) => pId.toString());
      dto.performers = performers.filter((p) => performerStringIds.includes(p._id.toString()));
    }
    if (categories.length) {
      const categoryStringIds = compact(video.categoryIds).map((pId) => pId.toString());
      dto.categories = categories.filter((c) => categoryStringIds.includes(c._id.toString()));
    }
    if (tags.length) {
      dto.tags = tags.map((tag) => tag.name);
    }
    const authenticated = true; // TODO - Check purchased videos for videos for sale or user is admin.
    const fileInfo = video.source === 'internal'
      && (await this.getPublicFileInfo(id, authenticated));
    if (fileInfo.main) dto.setMainVideo(fileInfo.main);
    return dto;
  }

  // TODO - define payload
  public async signUploadThumbnailUrl(
    payload?: Partial<SignedUploadRequestPayload>
  ) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'video-thumbnail',
      mediaType: 'image',
      filename: payload?.filename,
      acl: 'public-read'
      // TODO - define params for push queue or webhook to update video convert status if needed
      // params
    });
    return res;
  }

  public async signUploadMainVideoUrl(
    payload?: Partial<SignedUploadRequestPayload>
  ) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'video-main',
      mediaType: 'video',
      filename: payload?.filename,
      acl: 'authenticated-read',
      notification: {
        channel: VIDEO_FILE_CHANNEL,
        data: {}
      }
      // TODO - define params for push queue or webhook to update video convert status if needed
    });
    return res;
  }

  /**
   * @param videoId
   * @param authenticated
   * @param expiresIn
   */
  public async getPublicFileInfo(
    videoId: string | ObjectId,
    authenticated = false,
    expiresIn = 3600
  ) {
    const video = await this.findByIdOrSlug(videoId);
    if (!video || !video.fileId) return {} as any;

    const fileIds = [video.fileId.toString()];
    const publicInfo = await this.fileService.getFilesPublicInfo({
      fileIds,
      authenticated,
      expiresIn
    });
    if (!publicInfo[video.fileId.toString()]) return {};

    return {
      // TODO - check if video has multiple transcode files
      main: publicInfo[video.fileId.toString()]
    };
  }

  public async getFileUploadedInfo(fileId: string) {
    const file = await this.fileService.getPubicInfo({
      fileId
    });

    return file;
  }

  public async addView(videoId: string | ObjectId) {
    return this.VideoModel.updateOne(
      {
        _id: toObjectId(videoId)
      },
      {
        $inc: {
          'stats.views': 1
        }
      }
    );
  }

  public async updateStats(
    id: string | ObjectId,
    payload: Record<string, number>
  ) {
    return this.VideoModel.updateOne({ _id: id }, { $inc: payload });
  }

  // Calculate total views by performerId
  public async countTotalVideoViewsByPerformerId(
    performerId: ObjectId
  ): Promise<number> {
    const result = await this.VideoModel.aggregate([
      {
        // Query all documents containing performerId in the performerIds array
        $match: {
          performerIds: performerId
        }
      },
      {
        $group: {
          _id: null, // No need to group by any field
          totalViews: { $sum: '$stats.views' } // Calculate total views
        }
      }
    ]).allowDiskUse(true);

    // If no results are returned, it means that performerId has no videos.
    return result.length > 0 ? result[0].totalViews : 0;
  }

  public async countTotalVideoViewsByChannelId(
    channelIds: string[] | ObjectId[]
  ) {
    return this.VideoModel.aggregate([
      {
        $match: {
          channelId: {
            $in: channelIds.map((channelId) => toObjectId(channelId))
          }
        }
      },
      {
        $group: {
          _id: '$channelId',
          totalViews: { $sum: '$stats.views' },
          totalVideos: { $sum: 1 }
        }
      }
    ]).allowDiskUse(true);
  }
}
