import { Inject, Injectable, forwardRef } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { uniq } from 'lodash';
import { FilterQuery, Model } from 'mongoose';

import { REACTION_ACTION } from 'src/constants';
import {
  PageableData, populateDBSort, populateDbQuery
} from 'src/core';
import { ProfileDto, VideoDto } from 'src/dtos';
import { FileService } from 'src/file-module/services';
import { VideoSearchPayload } from 'src/payloads';
import { Video, VideoDocument } from 'src/schemas';

import { CategorySearchService } from '../category';
import { PerformerSearchService } from '../performer';
import { ProfileService } from '../profile';
import { ReactionService } from '../reaction';

@Injectable()
export class VideoSearchService {
  constructor(
    @InjectModel(Video.name) private readonly VideoModel: Model<VideoDocument>,
    @Inject(forwardRef(() => ReactionService))
    private readonly reactionService: ReactionService,
    private readonly categorySearchService: CategorySearchService,
    private readonly performerSearchService: PerformerSearchService,
    private readonly fileService: FileService,
    private readonly profileService: ProfileService
  ) { }

  public async populateVideoData(data: VideoDocument[], user?: ProfileDto): Promise<VideoDto[]> {
    const videoIds = data.map((v) => v._id.toString());
    const isAdmin = user && user.roles.includes('admin');

    let categoryIds = [];
    let performerIds = [];
    let createdBys = [];
    const fileIds = [];

    data.forEach((v) => {
      if (v.performerIds && v.performerIds.length) {
        performerIds = uniq(performerIds.concat(v.performerIds));
      }
      if (v.categoryIds && v.categoryIds.length) {
        categoryIds = uniq(categoryIds.concat(v.categoryIds));
      }
      if (v.fileId) {
        fileIds.push(v.fileId);
      }
      if (v.createdBy && isAdmin) {
        createdBys = uniq(createdBys.concat(v.createdBy));
      }
    });
    const [reactions, performers, categories, fileThumbnails, videos, creators] = await Promise.all([
      user ? this.reactionService.findByQuery({
        objectId: {
          $in: videoIds
        },
        createdBy: user._id
      }) : [],
      performerIds.length > 0 ? this.performerSearchService.findByIds(performerIds, user) : [],
      categoryIds.length > 0 ? this.categorySearchService.findByIds(categoryIds) : [],
      fileIds.length > 0 ? this.fileService.getListFilesThumbnail({
        fileIds,
        authenticated: true,
        expiresIn: 3600
      }) : [],
      fileIds.length > 0 && isAdmin ? this.fileService.getFilesPublicInfo({ fileIds, authenticated: true }) : {},
      createdBys.length > 0 ? this.profileService.findByIds(createdBys) : []
    ]);

    return data.map((v) => {
      const video = plainToInstance(VideoDto, v);
      const like = reactions.find((l) => l.objectId.toString() === v._id.toString() && l.action === REACTION_ACTION.LIKE);
      const bookmark = reactions.find((l) => l.objectId.toString() === v._id.toString() && l.action === REACTION_ACTION.BOOKMARK);

      video.isLiked = !!like;
      video.isBookmarked = !!bookmark;

      if (v.performerIds && v.performerIds.length) {
        const performerStringIds = video.performerIds.map((pId) => pId.toString());
        video.performers = performers.filter((p) => performerStringIds.includes(p._id.toString()));
      }
      if (v.categoryIds && v.categoryIds.length) {
        const categoryStringIds = v.categoryIds.filter((c) => !!c).map((pId) => pId.toString());
        video.categories = categories.filter((c) => categoryStringIds.includes(c._id.toString()));
      }
      if (fileThumbnails && v.fileId && fileThumbnails[v.fileId.toString()]) {
        video.thumbnails = [fileThumbnails[v.fileId.toString()]?.thumbnailUrl];
      }
      if (v.source === 'internal' && v.fileId && !!Object.getOwnPropertyDescriptor(videos, v.fileId.toString())) {
        video.setMainVideo(videos[v.fileId.toString()]);
      }
      if (v.createdBy && isAdmin) {
        const creator = creators.find((item) => item._id.toString() === v.createdBy.toString());
        video.setCreator(creator);
      }
      return video;
    });
  }

  public async advancedSearch(payload: VideoSearchPayload, user?: ProfileDto): Promise<PageableData<VideoDto>> {
    const populateQuery = populateDbQuery(payload, {
      equal: ['status'],
      objectId: ['createdBy'],
      boolean: ['isByMember'],
      inArray: ['performerIds', 'tags', 'categoryIds']
    });
    const query: FilterQuery<VideoDocument> = { ...populateQuery };

    if (payload.q) {
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      const searchValue = { $search: `"${payload.q.toLowerCase()}"` };
      query.$or = [
        {
          $text: searchValue
        },
        {
          title: { $regex: regexp }
        },
        {
          slug: { $regex: regexp }
        },
        {
          tags: payload.q.toLowerCase()
        }
      ];
    }

    const sort = populateDBSort(payload);

    const [data, total] = await Promise.all([
      this.VideoModel.find(query)
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.VideoModel.countDocuments(query)
    ]);

    return {
      data: await this.populateVideoData(data, user),
      total
    };
  }

  public async search(payload: VideoSearchPayload, user: ProfileDto): Promise<PageableData<VideoDto>> {
    const populateQuery = populateDbQuery(payload, {
      equal: ['status'],
      objectId: ['createdBy'],
      boolean: ['verified'],
      inArray: ['performerIds', 'tags', 'categoryIds'],
      excludeId: payload.excludeId
    });
    const query: FilterQuery<VideoDocument> = {
      // status: 'active'
      ...populateQuery
    };

    if (payload.q) {
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      const searchValue = { $regex: regexp };
      query.$or = [
        {
          title: searchValue
        },
        {
          slug: searchValue
        },
        {
          tags: payload.q.toLowerCase()
        }
      ];
    }


    const sort = [];
    switch (payload.sortBy) {
      case 'oldest':
        sort.push(['createdAt', 'asc']);
        break;
      case 'trending':
        sort.push(['stats.views', 'desc']);
        sort.push(['createdAt', 'desc']);
        break;
      case 'rating':
        sort.push(['stats.likes', 'desc']);
        sort.push(['createdAt', 'desc']);
        break;
      default:
        sort.push(['createdAt', 'desc']);
    }

    const [data, total] = await Promise.all([
      this.VideoModel.find(query)
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.VideoModel.countDocuments(query)
    ]);

    return {
      data: await this.populateVideoData(data, user),
      total
    };
  }
}
