import { Injectable, Logger } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model } from 'mongoose';

import { REACTION_ACTION } from 'src/constants';
import { ProfileDto, VideoDto } from 'src/dtos';
import { IGetReactionObjectInfoResponse, IUpdateObjectStats } from 'src/interfaces';
import { ReactionUpdateStatsPayload } from 'src/payloads';
import { Video, VideoDocument } from 'src/schemas';

import { VideoSearchService } from './video-search.service';
import { AbstractReaction } from '../reaction';

@Injectable()
export class VideoReactionService extends AbstractReaction {
  private logger = new Logger(VideoReactionService.name);

  constructor(
    @InjectModel(Video.name) private readonly VideoModel: Model<VideoDocument>,
    private readonly videoSearchService: VideoSearchService
  ) {
    super();
  }

  public async getReactionObjectInfo(ids: string[], user?: ProfileDto): Promise<IGetReactionObjectInfoResponse<VideoDto>> {
    const data = await this.VideoModel.find({ _id: { $in: ids } })
      .lean();

    return {
      items: await this.videoSearchService.populateVideoData(data, user) // TO-DO need to check current user purchased the video
    };
  }

  public async handleUpdateStats(objectId: string | ObjectId, payload: ReactionUpdateStatsPayload): Promise<IUpdateObjectStats> {
    try {
      const video = await this.VideoModel.findById(objectId);
      if (!video) {
        return;
      }
      const { action, value } = payload;
      let fieldToUpdate: string;
      switch (action) {
        case REACTION_ACTION.LIKE:
          fieldToUpdate = 'stats.likes';
          break;
        case REACTION_ACTION.FAVOURITE:
          fieldToUpdate = 'stats.favourites';
          break;
        case REACTION_ACTION.DISLIKE:
          fieldToUpdate = 'stats.dislikes';
          break;
        default:
          break;
      }

      if (!fieldToUpdate) return;

      const updateQuery: FilterQuery<VideoDocument> = {
        _id: objectId
      };
      if (value < 0) {
        updateQuery[fieldToUpdate] = { $gte: 1 };
      }
      await this.VideoModel.updateOne(updateQuery, {
        $inc: {
          [fieldToUpdate]: value
        }
      });
    } catch (e) {
      this.logger.error(e);
    }
  }
}
