import { HttpException, Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { merge } from 'lodash';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import { EVENT } from 'src/constants';
import {
  EntityNotFoundException, QueueMessageService, createAlias, isObjectId
} from 'src/core';
import { TagDto, ProfileDto } from 'src/dtos';
import { TagCreatePayload, TagUpdatePayload } from 'src/payloads';
import { Tag, TagDocument } from 'src/schemas';

@Injectable()
export class TagService {
  constructor(
    @InjectModel(Tag.name) private readonly TagModel: Model<TagDocument>,
    private readonly queueMessageService: QueueMessageService
  ) { }

  public async create(payload: TagCreatePayload, current: ProfileDto): Promise<TagDto> {
    const key = payload.key.toLowerCase();
    const existKey = await this.TagModel.findOne({ key });
    if (existKey) {
      throw new HttpException('This key has been taken, please choose another one', 400);
    }
    const tag = await this.TagModel.create({
      ...payload,
      createdBy: current._id
    });

    return plainToInstance(TagDto, tag.toObject());
  }

  public async findByIdOrKey(id: string | ObjectId): Promise<TagDocument> {
    const query = id instanceof ObjectId || isObjectId(id)
      ? { _id: id }
      : { slug: id };
    return this.TagModel.findOne(query);
  }

  public async update(id: string | ObjectId, payload: TagUpdatePayload): Promise<TagDto> {
    const tag = await this.TagModel.findById(id);
    if (!tag) throw new EntityNotFoundException();

    const key = payload.key && payload.key.toLowerCase();
    if (key && key !== tag.key) {
      const existKey = await this.TagModel.findOne({ key });
      if (existKey) {
        throw new HttpException('This key has been taken, please choose another one', 400);
      }
    }

    merge(tag, payload);
    tag.updatedAt = new Date();
    await tag.save();
    this.queueMessageService.publish('VIDEO_TAG_CHANNEL', {
      data: tag,
      eventName: EVENT.UPDATED
    });

    return plainToInstance(TagDto, tag.toObject());
  }

  public async delete(id: string | ObjectId): Promise<any> {
    const tag = await this.TagModel.findById(id);
    if (!tag) throw new EntityNotFoundException();

    await tag.deleteOne();
    this.queueMessageService.publish('VIDEO_TAG_CHANNEL', {
      data: tag,
      eventName: EVENT.DELETED
    });
    return true;
  }


  public async getDetails(id: string | ObjectId): Promise<TagDto> {
    const tag = await this.findByIdOrKey(id);
    if (!tag) throw new EntityNotFoundException();

    const dto = plainToInstance(TagDto, tag.toObject());
    return dto;
  }

  public async findByIds(ids: ObjectId[]) {
    return this.TagModel.find({ _id: { $in: ids }, status: 'active' });
  }

  public async createTagViaVideoImport(tags: string | string[]) {
    // eslint-disable-next-line no-nested-ternary
    const arrayTagsNames = Array.isArray(tags) ? tags : tags.includes(',') ? tags.split(',') : tags.split(';');
    const ids = [];
    if (arrayTagsNames.length > 0) {
      await arrayTagsNames.reduce(async (lp, t) => {
        await lp;

        const key = createAlias(t.trim());
        let tag = await this.TagModel.findOne({ key });

        if (!tag) {
          tag = await this.TagModel.create({
            name: t.trim(),
            key
          });
        }

        ids.push(tag.key);

        return Promise.resolve();
      }, Promise.resolve());
    }

    return ids;
  }

  public async createFromList(tags: string[] | ObjectId[]) {
    const tagsToCreate = tags.filter((t) => t.toString().trim() && !isObjectId(t.toString()));
    if (!tagsToCreate) return;
    await tagsToCreate.reduce(async (res, tag) => {
      await res;
      const key = createAlias(tag.toString().trim());
      const existKey = await this.TagModel.findOne({ key });
      if (existKey) return Promise.resolve();
      await this.TagModel.create({
        key,
        name: tag.toString().trim()
      });
      return Promise.resolve();
    }, Promise.resolve());
  }

  public async findFromList(tags: string[] | ObjectId[]) {
    const tagIds = tags.filter((t) => isObjectId(t.toString()));
    const tagKeys = tags.filter((t) => !!t.toString().trim() && !isObjectId(t.toString())).map((t) => createAlias(t.toString().trim()));
    if (!tagIds.length && !tagKeys.length) return [];
    const query: Record<string, any> = { $or: [] };
    if (tagIds.length) query.$or.push({ _id: { $in: tagIds } });
    if (tagKeys.length) query.$or.push({ key: { $in: tagKeys } });

    return this.TagModel.find(query);
  }
}
