import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { FilterQuery, Model } from 'mongoose';

import { PageableData, populateDBSort } from 'src/core';
import { TagDto } from 'src/dtos';
import { TagSearchPayload } from 'src/payloads';
import { Tag, TagDocument } from 'src/schemas';


@Injectable()
export class TagSearchService {
  constructor(
    @InjectModel(Tag.name) private readonly TagModel: Model<TagDocument>
  ) { }

  public async advancedSearch(payload: TagSearchPayload): Promise<PageableData<TagDto>> {
    const query: FilterQuery<TagDocument> = {
    };
    if (payload.status) query.status = payload.status;
    if (payload.q) {
      query.$or = [
        {
          name: { $regex: payload.q }
        },
        {
          key: { $regex: payload.q }
        }
      ];
    }

    const sort = populateDBSort(payload);

    const [data, total] = await Promise.all([
      this.TagModel.find(query)
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.TagModel.countDocuments(query)
    ]);

    return {
      data: data.map((tag) => plainToInstance(TagDto, tag)),
      total
    };
  }

  public async search(payload: TagSearchPayload): Promise<PageableData<Partial<TagDto>>> {
    const query: FilterQuery<TagDocument> = {
      status: 'active'
    };

    if (payload.q) {
      query.$or = [
        {
          name: { $regex: payload.q }
        },
        {
          key: { $regex: payload.q }
        }
      ];
    }

    const [data, total] = await Promise.all([
      this.TagModel.find(query)
        .sort({ [payload.sortBy || 'createdAt']: 'desc' })
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.TagModel.countDocuments(query)
    ]);

    return {
      data: data.map((tag) => plainToInstance(TagDto, tag).toPublicResponse()),
      total
    };
  }
}
