import { Injectable, HttpException } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { Model } from 'mongoose';

import { EntityNotFoundException, QueueMessageService } from 'src/core';
import { SettingDto } from 'src/dtos';
import { SignedUploadRequestPayload } from 'src/file-module/payloads';
import { FileService } from 'src/file-module/services';
import { SettingCreatePayload, SettingUpdatePayload } from 'src/payloads';
import { Setting, SettingDocument } from 'src/schemas';

import { SETTING_CHANNEL, SETTING_FILE_CHANNEL } from '../constants';

@Injectable()
export class SettingService {
  static _settingCache = {} as Map<string, any>;

  // key and value
  static _publicSettingsCache = {} as any;

  constructor(
    @InjectModel(Setting.name) private readonly SettingModel: Model<SettingDocument>,
    private readonly queueMessageService: QueueMessageService,
    private readonly fileService: FileService
  ) {
    this.queueMessageService.subscribe(SETTING_CHANNEL, 'HANDLE_SETTINGS_CHANGE', this.subscribeChange.bind(this));
  }

  private async publishChange(setting: SettingDto) {
    await this.queueMessageService.publish(SETTING_CHANNEL, {
      eventName: 'update',
      data: setting
    });
  }

  private async subscribeChange() {
    // TODO - update directly to static variable?
    await this.syncCache();
  }

  public async syncCache(): Promise<void> {
    const settings = await this.SettingModel.find();
    settings.forEach((setting) => {
      const dto = plainToInstance(SettingDto, setting);
      SettingService._settingCache[dto.key] = dto;
      if (dto.visible && dto.public) {
        SettingService._publicSettingsCache[dto.key] = dto.value;
      }
    });
  }

  async get(key: string): Promise<SettingDto> {
    if (SettingService._settingCache[key]) {
      return SettingService._settingCache[key];
    }

    // TODO - handle events when settings change and reupdate here
    const data = await this.SettingModel.findOne({ key });
    if (!data) {
      return null;
    }
    const dto = plainToInstance(SettingDto, data);
    SettingService._settingCache[key] = dto;
    return dto;
  }

  async getKeyValue(key: string): Promise<any> {
    if (SettingService._settingCache[key]) {
      return SettingService._settingCache[key].value;
    }

    // TODO - handle events when settings change and reupdate here
    const data = await this.SettingModel.findOne({ key });
    if (!data) {
      return null;
    }
    const dto = plainToInstance(SettingDto, data);
    SettingService._settingCache[key] = dto;
    return dto.value;
  }

  async create(data: SettingCreatePayload): Promise<any> {
    const setting = await this.get(data.key);
    if (setting) {
      throw new HttpException('Setting key exist', 400);
    }

    // reupdate the setting list
    // TODO - must publish and subscribe to redis channel, so all instances (if run multiple)
    // have the same data
    await this.syncCache();
    return this.SettingModel.create(data);
  }

  async update(key: string, data: SettingUpdatePayload): Promise<SettingDto> {
    const setting = await this.SettingModel.findOne({ key });
    if (!setting) {
      throw new EntityNotFoundException();
    }
    if (data.description) setting.set('description', data.description);
    if (data.name) setting.set('name', data.name);
    setting.set('value', data.value);
    await setting.save();
    const dto = plainToInstance(SettingDto, setting);
    await this.publishChange(dto);
    return dto;
  }

  // get public and visible settings
  async getPublicSettings(): Promise<Record<string, any>> {
    return SettingService._publicSettingsCache;
  }

  async getAutoloadPublicSettingsForUser(): Promise<Record<string, any>> {
    const autoloadSettings = {} as any;
    return Object.keys(SettingService._settingCache).reduce((settings, key) => {
      const results = settings;
      if (SettingService._settingCache[key].autoload && SettingService._publicSettingsCache[key]) {
        results[key] = SettingService._settingCache[key].value;
      }
      return results;
    }, autoloadSettings);
  }

  getPublicValueByKey(key: string) {
    return {
      value: SettingService._publicSettingsCache[key]?.value || null
    };
  }

  getPublicValueByKeys(keys: string[]) {
    return keys.reduce((lp, key) => {
      const results = lp;
      results[key] = SettingService._publicSettingsCache[key];
      return results;
    }, {} as any);
  }

  async getCommissionSettings() {
    return this.SettingModel.find({ group: 'commission' });
  }

  /**
   * get all settings which are editable
   */
  async getEditableSettings(group?: string): Promise<SettingDto[]> {
    const query = { editable: true } as any;
    if (group) {
      query.group = group;
    }
    // custom sort odering
    const settings = await this.SettingModel.find(query).sort({ ordering: 'asc' });

    return settings.map((s) => plainToInstance(SettingDto, s.toObject()));
  }

  public static getByKey(key: string) {
    return SettingService._settingCache[key] || null;
  }

  public static getValueByKey(key: string) {
    return SettingService._settingCache[key] ? SettingService._settingCache[key].value : null;
  }

  public async getSignedUploadImageFile(payload?: Partial<SignedUploadRequestPayload>) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'setting-image',
      mediaType: 'image',
      filename: payload?.filename,
      acl: 'public-read',
      notification: {
        channel: SETTING_FILE_CHANNEL,
        data: payload
      }
    });
    return res;
  }

  public async getFileUploadedInfo(fileId: string) {
    const file = await this.fileService.getPubicInfo({
      fileId
    });

    return file;
  }
}
