import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model } from 'mongoose';

import { EVENT, REACTION_ACTION, REACTION_CHANNEL } from 'src/constants';
import { EntityNotFoundException, QueueMessageService } from 'src/core';
import { ReactionDto, ProfileDto } from 'src/dtos';
import { ReactionInfo } from 'src/interfaces';
import { ReactDeletePayload, ReactionCreatePayload } from 'src/payloads';
import { Reaction, ReactionDocument } from 'src/schemas';

@Injectable()
export class ReactionService {
  constructor(
    @InjectModel(Reaction.name) private readonly ReactionModel: Model<ReactionDocument>,
    private readonly queueService: QueueMessageService
  ) { }

  public async create(payload: ReactionCreatePayload, user: ProfileDto): Promise<ReactionDto> {
    let reaction = await this.ReactionModel.findOne({
      objectType: payload.objectType,
      objectId: payload.objectId,
      createdBy: user._id,
      action: payload.action
    });
    if (reaction) {
      return plainToInstance(ReactionDto, reaction.toObject());
    }
    reaction = await this.ReactionModel.create({
      ...payload,
      createdBy: user._id
    });

    const rDto = plainToInstance(ReactionDto, reaction.toObject());
    await this.queueService.publish(REACTION_CHANNEL, {
      eventName: EVENT.CREATED,
      data: rDto
    });
    return rDto;
  }

  public async delete(payload: ReactDeletePayload, user: ProfileDto): Promise<any> {
    const reaction = await this.ReactionModel.findOne({
      objectType: payload.objectType,
      objectId: payload.objectId,
      createdBy: user._id,
      action: payload.action
    });
    if (!reaction) throw new EntityNotFoundException();

    await reaction.deleteOne();
    const rDto = plainToInstance(ReactionDto, reaction.toObject());
    await this.queueService.publish(REACTION_CHANNEL, {
      eventName: EVENT.DELETED,
      data: rDto
    });
    return true;
  }

  public async findByQuery(payload: Record<string, any>) {
    const query: FilterQuery<ReactionDocument> = {
      ...payload
    };
    return this.ReactionModel.find(query);
  }

  public async isLiked(objectId: string | ObjectId, user: ProfileDto) {
    const reaction = await this.ReactionModel.findOne({
      objectId,
      createdBy: user._id,
      action: REACTION_ACTION.LIKE
    });

    return !!reaction;
  }

  public async isBookmarked(objectId: string | ObjectId, user: ProfileDto) {
    const reaction = await this.ReactionModel.findOne({
      objectId,
      createdBy: user._id,
      action: REACTION_ACTION.BOOKMARK
    });

    return !!reaction;
  }

  public async isFavourite(objectId: string | ObjectId, user: ProfileDto) {
    const reaction = await this.ReactionModel.findOne({
      objectId,
      createdBy: user._id,
      action: REACTION_ACTION.FAVOURITE
    });

    return !!reaction;
  }

  public async getReactionInfo(objectId: string | ObjectId, user: ProfileDto): Promise<ReactionInfo> {
    const reactions = await this.ReactionModel.find({
      objectId,
      createdBy: user._id
    });

    const result: ReactionInfo = {
      isLiked: false,
      isBookmarked: false,
      isFollowed: false,
      isDisLiked: false
    };
    reactions.forEach((r) => {
      switch (r.action) {
        case REACTION_ACTION.LIKE:
          result.isLiked = true;
          break;
        case REACTION_ACTION.BOOKMARK:
          result.isBookmarked = true;
          break;
        case REACTION_ACTION.FOLLOW:
          result.isFollowed = true;
          break;
        case REACTION_ACTION.DISLIKE:
          result.isDisLiked = true;
          break;
        default:
          break;
      }
    });

    return result;
  }
}
