import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { FilterQuery, Model } from 'mongoose';

import { PageableData } from 'src/core';
import { ProfileDto, ReactionDto } from 'src/dtos';
import { ReactionSearchRequestPayload } from 'src/payloads';
import { Reaction, ReactionDocument } from 'src/schemas';

import { ReactionObjectService } from './reaction-object.service';


@Injectable()
export class ReactionSearchService {
  constructor(
    @InjectModel(Reaction.name) private readonly ReactionModel: Model<ReactionDocument>,
    private readonly reactionObjectService: ReactionObjectService
  ) { }

  public async advancedSearch(payload: ReactionSearchRequestPayload): Promise<PageableData<ReactionDto>> {
    const query: FilterQuery<ReactionDocument> = {};

    if (payload.q) {
      query.$or = [
        {
          name: { $regex: payload.q }
        }
      ];
    }

    const [data, total] = await Promise.all([
      this.ReactionModel.find(query)
        .sort({ [payload.sortBy || 'createdAt']: 'desc' })
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.ReactionModel.countDocuments(query)
    ]);

    return {
      data: data.map((reaction) => plainToInstance(ReactionDto, reaction)),
      total
    };
  }

  public async search(payload: ReactionSearchRequestPayload, user: ProfileDto): Promise<PageableData<ReactionDto>> {
    const { objectType, action } = payload;
    const query: FilterQuery<ReactionDocument> = {
      createdBy: user._id,
      objectType,
      action
    };

    const [data, total] = await Promise.all([
      this.ReactionModel.find(query)
        .sort({ [payload.sortBy || 'createdAt']: 'desc' })
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.ReactionModel.countDocuments(query)
    ]);

    const objectIds = data.map((item) => item.objectId);

    const objectInfo = await this.reactionObjectService.getObjectInfo(payload.objectType, objectIds, user);
    const { items } = objectInfo;

    const dto = data.map((reaction) => {
      const reactionDto = plainToInstance(ReactionDto, reaction);
      reactionDto.objectInfo = items.find((objectItem) => objectItem._id.toString() === reaction.objectId.toString());
      return reactionDto;
    });

    return {
      data: dto,
      total
    };
  }
}
