import { BadRequestException, HttpException, Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model } from 'mongoose';

import { EVENT, PROFILE_CHANNEL } from 'src/constants';
import {
  EntityNotFoundException, QueueMessageService, isInvalidUsername, isObjectId
} from 'src/core';
import { ProfileDto } from 'src/dtos';
import {
  EmailHasBeenTakenException, EmailNotFoundException, UsernameExistedException, UsernameNotFoundException
} from 'src/exceptions';
import { FileService } from 'src/file-module/services';
import { AdminProfileUpdatePayload, ProfileCreatePayload, ProfileUpdatePayload } from 'src/payloads';
import { Profile } from 'src/schemas';


@Injectable()
export class ProfileService {
  constructor(
    @InjectModel(Profile.name) private readonly ProfileModel: Model<Profile>,
    private readonly queueMessageService: QueueMessageService,
    private readonly fileService: FileService
  ) { }

  /**
   * find user by username
   * @param username
   * @returns
   */
  async findByUsername(username: string): Promise<ProfileDto> {
    const usernameLowercase = username.toLowerCase();
    const profile = await this.ProfileModel.findOne({ username: usernameLowercase });
    if (!profile) throw new UsernameNotFoundException();
    const dto = plainToInstance(ProfileDto, profile.toObject());
    return dto;
  }

  /**
   * find user by email
   * @param username
   * @returns
   */
  async findByEmail(email: string): Promise<ProfileDto> {
    const profile = await this.ProfileModel.findOne({
      email: email.toLowerCase()
    });
    if (!profile) throw new EmailNotFoundException();
    const dto = plainToInstance(ProfileDto, profile.toObject());
    return dto;
  }

  /**
   * find user by id
   * @param username
   * @returns
   */
  async findById(id: ObjectId | string): Promise<ProfileDto> {
    // TODO - validate id object
    const profile = await this.ProfileModel.findOne({
      _id: id
    });
    if (!profile) throw new EntityNotFoundException();
    const dto = plainToInstance(ProfileDto, profile.toObject());

    if (!profile.avatar && profile.avatarId) {
      const file = await this.fileService.getPubicInfo({ fileId: profile.avatarId, authenticated: true });
      if (file) dto.avatar = file.url;
    }

    return dto;
  }

  /**
   * get profile details with related data
   * @param id
   * @param fromProfile
   * @returns
   */
  async getDetails(id: string): Promise<ProfileDto> {
    const query: FilterQuery<Profile> = isObjectId(id) ? { _id: id } : { username: id };
    const profile = await this.ProfileModel.findOne(query);
    if (!profile) {
      throw new EntityNotFoundException();
    }
    const dto = plainToInstance(ProfileDto, profile.toObject());

    if (!profile.avatar && profile.avatarId) {
      const file = await this.fileService.getPubicInfo({ fileId: profile.avatarId, authenticated: true });
      if (file) dto.avatar = file.url;
    }

    return dto;
  }

  public async findByIds(ids: string[] | ObjectId[]) {
    const profiles = await this.ProfileModel.find({
      _id: { $in: ids }
    });
    return profiles.map((profile) => plainToInstance(ProfileDto, profile.toObject()));
  }

  /**
   * create new profile with unique username, email
   * @param payload
   * @returns
   */
  async create(payload: ProfileCreatePayload): Promise<ProfileDto> {
    if (isInvalidUsername(payload.username)) {
      throw new HttpException({
        message: 'Invalid username!'
      }, 400);
    }

    // check unique value
    // TODO - check if need to use validator
    const username = payload.username.toLowerCase();
    const email = payload.email.toLowerCase();
    const [existingUsername, existingEmail] = await Promise.all([
      this.ProfileModel.findOne({
        username
      }),
      this.ProfileModel.findOne({
        email
      })
    ]);

    if (existingUsername) {
      throw new UsernameExistedException();
    }
    if (existingEmail) {
      throw new EmailHasBeenTakenException();
    }

    const dataToCreate = {
      ...payload,
      email,
      username
    };

    const newProfile = await this.ProfileModel.create(dataToCreate);
    const dto = plainToInstance(ProfileDto, newProfile.toObject());

    await this.queueMessageService.publish(PROFILE_CHANNEL, {
      eventName: EVENT.CREATED,
      channel: PROFILE_CHANNEL,
      data: dto
    });

    return dto;
  }

  async update(id: ObjectId | string, payload: Partial<AdminProfileUpdatePayload>): Promise<ProfileDto> {
    if (typeof id === 'string' && !isObjectId(id)) {
      throw new BadRequestException();
    }

    let profile = await this.ProfileModel.findById(id);
    if (!profile) {
      throw new EntityNotFoundException();
    }

    if (isInvalidUsername(payload.username)) {
      throw new HttpException({
        message: 'Invalid username!'
      }, 400);
    }

    const data = { ...payload } as any;

    const username = (data.username && data.username.toLowerCase()) || '';

    const email = (data.email && data.email.toLowerCase()) || '';
    const [existingUsername, existingEmail] = await Promise.all([
      this.ProfileModel.findOne({
        username,
        _id: {
          $ne: id
        }
      }),
      this.ProfileModel.findOne({
        email,
        _id: {
          $ne: id
        }
      })
    ]);
    if (existingUsername) {
      throw new UsernameExistedException();
    }
    if (existingEmail) {
      throw new EmailHasBeenTakenException();
    }

    if (data.dateOfBirth) {
      data.dateOfBirth = new Date(data.dateOfBirth);
    }
    if (data.username) {
      data.username = data.username.toLowerCase();
    }
    if (data.email) {
      data.email = data.email.toLowerCase();
    }

    data.updatedAt = new Date();
    // data.hashTags = payload.bio ? findHashTags(payload.bio) : [];
    await this.ProfileModel.updateOne({ _id: id }, data);

    profile = await this.ProfileModel.findById(id);
    const dto = plainToInstance(ProfileDto, profile.toObject());

    await this.queueMessageService.publish(PROFILE_CHANNEL, {
      eventName: EVENT.UPDATED,
      channel: PROFILE_CHANNEL,
      data: dto
    });

    return dto;
  }

  public async selfUpdate(profileId: string | ObjectId, payload: ProfileUpdatePayload) {
    return this.update(profileId, payload);
  }
}
