import {
  HttpException,
  Injectable
} from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { InjectModel } from '@nestjs/mongoose';
import { Model, ObjectId } from 'mongoose';

import { randomString } from 'src/core';
import { Profile, ProfileVerification } from 'src/schemas';
import { MailService } from 'src/services/mailer/mail.service';

import { ProfileService } from './profile.service';

@Injectable()
export class ProfileVerificationService {
  constructor(
    private configService: ConfigService,
    private profileService: ProfileService,
    private mailService: MailService,
    @InjectModel(Profile.name) private readonly ProfileModel: Model<Profile>,
    @InjectModel(ProfileVerification.name) private readonly ProfileVerificationModel: Model<ProfileVerification>
  ) { }

  public async sendEmailActiveModel(profileId: string | ObjectId, email: string) {
    const verification = await this.ProfileVerificationModel.findOne({
      type: 'email',
      profileId,
      value: email.toLowerCase()
    });
    if (verification?.verified) return true;

    const profile = await this.profileService.findById(profileId.toString());
    // do not throw error
    if (!profile) return true;
    await this.mailService.sendMail({
      to: email,
      template: 'active-account-model',
      data: {
        profile
      }
    });
    return true;
  }

  public async sendEmailVerification(profileId: string | ObjectId, email: string) {
    const verification = await this.ProfileVerificationModel.findOne({
      type: 'email',
      profileId,
      value: email.toLowerCase()
    });
    if (verification?.verified) return true;

    const profile = await this.profileService.findById(profileId.toString());
    // do not throw error
    if (!profile) return true;

    const token = randomString(15);
    if (!verification) {
      await this.ProfileVerificationModel.create({
        type: 'email',
        profileId,
        value: email.toLowerCase(),
        token,
        verified: false,
        createdAt: new Date(),
        updatedAt: new Date()
      });
    } else {
      await this.ProfileVerificationModel.updateOne({
        _id: verification._id
      }, {
        $set: {
          token,
          verified: false,
          createdAt: new Date(),
          updatedAt: new Date()
        }
      });
    }

    const verificationLink = new URL(this.configService.get('EMAIL_VERIFICATION_PAGE_URL'));
    verificationLink.searchParams.append('token', token);
    return true;
  }

  public async verifyEmail(profileId: string | ObjectId, email: string, verified: boolean = true) {
    const verification = await this.ProfileVerificationModel.findOne({
      type: 'email',
      profileId,
      value: email.toLowerCase()
    });

    if (!verification) {
      await this.ProfileVerificationModel.create({
        type: 'email',
        profileId,
        value: email.toLowerCase(),
        token: null,
        verified,
        createdAt: new Date(),
        updatedAt: new Date()
      });
    } else {
      await this.ProfileVerificationModel.updateOne({
        _id: verification._id
      }, {
        $set: {
          token: null,
          verified,
          updatedAt: new Date()
        }
      });
      await this.profileService.update(profileId as any, { verifiedEmail: true } as any);
    }

    return true;
  }

  public async verifyEmailByToken(token: string) {
    const verification = await this.ProfileVerificationModel.findOne({
      token: token.toString()
    });
    // TODO - recheck in 24h?

    if (!verification) {
      throw new HttpException({
        message: 'Invalid token'
      }, 400);
    } else {
      await this.ProfileVerificationModel.updateOne({
        _id: verification._id
      }, {
        $set: {
          token: null,
          verified: true,
          updatedAt: new Date()
        }
      });

      await this.ProfileModel.updateOne({ _id: verification.profileId }, { $set: { verifiedEmail: true, status: 'active' } });
    }

    return true;
  }
}
