import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { FilterQuery, Model } from 'mongoose';

import { PageableData } from 'src/core';
import { ProfileDto } from 'src/dtos';
import { ProfileAdvancedSearchPayload } from 'src/payloads';
import { Profile, ProfileDocument } from 'src/schemas';


@Injectable()
export class ProfileSearchService {
  constructor(
    @InjectModel(Profile.name) private readonly ProfileModel: Model<ProfileDocument>
  ) { }

  public async advancedSearch(payload: ProfileAdvancedSearchPayload): Promise<PageableData<ProfileDto>> {
    const query: FilterQuery<ProfileDocument> = {};

    if (payload.q) {
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          name: { $regex: regexp }
        },
        {
          username: { $regex: regexp }
        },
        {
          email: { $regex: regexp }
        }
      ];
    }

    ['status'].forEach((field) => {
      if (typeof payload[field] !== 'undefined') {
        query[field] = payload[field];
      }
    });

    const sort = {
      [payload.sortBy || 'createdAt']: payload.sort || 1
    } as Record<string, any>;

    const [data, total] = await Promise.all([
      this.ProfileModel.find(query)
        .collation({ locale: 'en' })
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.ProfileModel.countDocuments(query)
    ]);

    return {
      data: data.map((profile) => plainToInstance(ProfileDto, profile)),
      total
    };
  }
}
