import { Injectable, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { InjectModel } from '@nestjs/mongoose';
import { Model, ObjectId } from 'mongoose';

import {
  AVATAR_CHANNEL, IAvatarPostback, PROFILE_CHANNEL, TOPIC_UPDATE_PROFILE_AVATAR_ON_UPLOADED
} from 'src/constants';
import { QueueEvent, QueueMessageService } from 'src/core';
import { PresignedUploadUrl } from 'src/file-module/interfaces';
import { FileService } from 'src/file-module/services';
import { Profile } from 'src/schemas';

@Injectable()
export class AvatarService implements OnModuleInit {
  constructor(
    @InjectModel(Profile.name) private readonly ProfileModel: Model<Profile>,
    private readonly fileService: FileService,
    private readonly queueMessageService: QueueMessageService,
    private readonly configService: ConfigService
  ) { }

  onModuleInit() {
    this.listenQueueMessage();
  }

  async listenQueueMessage() {
    await this.queueMessageService.subscribe(AVATAR_CHANNEL, TOPIC_UPDATE_PROFILE_AVATAR_ON_UPLOADED, this.handleAvatarUploaded.bind(this));
  }

  private async handleAvatarUploaded({ data }: QueueEvent<IAvatarPostback>) {
    /**
     * sample file data
     * {
        eventName: 'file_uploaded',
        data: {
          postbackData: { profileId: '64db353c807e14f51ca6b0b6' },
          file: {
            _id: '64edc0bfd9b3322dd6d83232',
            status: 'active',
            height: 1024,
            url: 'https://abc.com/image/upload/v1693303097/64edc0bfd9b3322dd6d83232.jpg',
          }
        }
      }
     */
    const { file, postbackData, status } = data.data;
    const { url: avatarUrl, _id: avatarId } = file;
    const { profileId } = postbackData;
    if (['processed', 'success'].includes(status)) {
      await this.ProfileModel.updateOne({ _id: profileId }, {
        $set: {
          avatar: avatarUrl,
          avatarId,
          updatedAt: new Date()
        }
      });
    }

    await this.queueMessageService.publish(PROFILE_CHANNEL, {
      eventName: 'updated',
      channel: PROFILE_CHANNEL,
      data: profileId
    });
    // TODO - handle call delete file if error
  }

  async getDirectUploadUrl(profileId?: string | ObjectId): Promise<PresignedUploadUrl> {
    // TODO - check valid profile?
    const avatarSize = this.configService.get('AVATAR_SIZE');
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'avatar',
      uploaderId: profileId && profileId.toString(),
      acl: 'public-read',
      mediaType: 'image',
      notification: {
        channel: AVATAR_CHANNEL,
        data: {
          profileId: profileId && profileId.toString()
        }
      },
      params: {
        createThumbnail: false,
        createBlurImage: false,
        forceImageWidth: avatarSize.w,
        forceImageHeight: avatarSize.h
      }
    } as any);
    return res;
  }
}
