import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { merge } from 'lodash';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import { EntityNotFoundException, createAlias, isObjectId } from 'src/core';
import { PostDto, ProfileDto } from 'src/dtos';
import { SignedUploadRequestPayload } from 'src/file-module/payloads';
import { FileService } from 'src/file-module/services';
import { PostCreatePayload, PostUpdatePayload } from 'src/payloads';
import { Post, PostDocument } from 'src/schemas';

@Injectable()
export class PostService {
  constructor(
    @InjectModel(Post.name) private readonly PostModel: Model<PostDocument>,
    private readonly fileService: FileService
  ) { }

  public async generateSlug(title: string, id?: string | ObjectId) {
    // consider if need unique slug with post type
    const slug = createAlias(title);
    const query = { slug } as any;
    if (id) {
      query._id = { $ne: id };
    }
    const count = await this.PostModel.countDocuments(query);
    if (!count) {
      return slug;
    }

    return this.generateSlug(`${slug}1`, id);
  }

  public async create(payload: PostCreatePayload, current: ProfileDto): Promise<PostDto> {
    const slug = await this.generateSlug(payload?.slug || payload.title);
    const post = await this.PostModel.create({
      ...payload,
      slug,
      createdBy: current._id
    });

    return plainToInstance(PostDto, post.toObject());
  }

  public async findByIdOrSlug(id: string | ObjectId): Promise<PostDocument> {
    const query = id instanceof ObjectId || isObjectId(id)
      ? { _id: id }
      : { slug: id };
    return this.PostModel.findOne(query);
  }

  public async update(id: string | ObjectId, payload: PostUpdatePayload): Promise<PostDto> {
    const post = await this.findByIdOrSlug(id);
    if (!post) throw new EntityNotFoundException();
    const slug = await this.generateSlug(payload?.slug || payload.title, post._id);
    merge(post, { ...payload, slug });
    post.categoryIds = payload?.categoryIds || [];
    post.updatedAt = new Date();
    await post.save();
    return plainToInstance(PostDto, post.toObject());
  }

  public async delete(id: string | ObjectId): Promise<any> {
    const post = await this.findByIdOrSlug(id);
    if (!post) throw new EntityNotFoundException();

    await post.deleteOne();
    return true;
  }


  public async getDetails(id: string | ObjectId) {
    const post = await this.findByIdOrSlug(id);
    if (!post) throw new EntityNotFoundException();

    const dto = plainToInstance(PostDto, post.toObject());
    return dto;
  }

  public async signUploadImageUrl(payload?: Partial<SignedUploadRequestPayload>) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'post-image',
      mediaType: 'image',
      filename: payload?.filename,
      acl: 'public-read'
    });
    return res;
  }

  public async getFileUploadedInfo(fileId: string, authenticated = false, expiresIn = 3600) {
    const file = await this.fileService.getPubicInfo({
      fileId,
      authenticated,
      expiresIn
    });
    if (!file) {
      throw new EntityNotFoundException();
    }
    return file;
  }
}
