import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { FilterQuery, Model } from 'mongoose';

import { PageableData, populateDBSort } from 'src/core';
import { PostDto } from 'src/dtos';
import { PostSearchPayload } from 'src/payloads';
import { Post, PostDocument } from 'src/schemas';


@Injectable()
export class PostSearchService {
  constructor(
    @InjectModel(Post.name) private readonly PostModel: Model<PostDocument>
  ) { }

  public async advancedSearch(payload: PostSearchPayload): Promise<PageableData<PostDto>> {
    const query: FilterQuery<PostDocument> = {};

    if (payload.q) {
      const searchValue = { $search: `"${payload.q.toLowerCase()}"` };
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          $text: searchValue
        },
        {
          title: { $regex: regexp }
        },
        {
          slug: { $regex: regexp }
        }
      ];
    }

    if (payload.status) {
      query.status = payload.status;
    }

    const sort = populateDBSort(payload);

    const [data, total] = await Promise.all([
      this.PostModel.find(query)
        .collation({ locale: 'en' })
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.PostModel.countDocuments(query)
    ]);

    return {
      data: data.map((post) => plainToInstance(PostDto, post)),
      total
    };
  }

  public async search(payload: PostSearchPayload): Promise<PageableData<PostDto>> {
    const query: FilterQuery<PostDocument> = {
      status: 'active'
    };

    if (payload.q) {
      query.$or = [
        {
          title: { $regex: payload.q }
        }
      ];
    }

    const [data, total] = await Promise.all([
      this.PostModel.find(query)
        .sort({ [payload.sortBy || 'createdAt']: 'desc' })
        .populate('categories')
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.PostModel.countDocuments(query)
    ]);

    return {
      data: data.map((post) => plainToInstance(PostDto, post)),
      total
    };
  }

  public async getMenus() {
    const data = await this.PostModel.aggregate([
      {
        $match: {
          status: 'active'
        }
      },
      {
        $unwind: '$categoryIds'
      },
      {
        $group: {
          _id: '$categoryIds',
          posts: { $push: { title: '$title', slug: '$slug', _id: '$_id' } }
        }
      },
      {
        $lookup: {
          from: 'categories',
          localField: '_id',
          foreignField: '_id',
          as: 'category'
        }
      },
      {
        $project: {
          otherField: 1,
          posts: 1,
          category: {
            // $first: '$category'
            $arrayElemAt: ['$category', 0]
          }
        }
      },
      {
        $sort: {
          'category.ordering': 1
        }
      }
    ]).allowDiskUse(true);

    return data;
  }
}
