import { Inject, Injectable, forwardRef } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { merge } from 'lodash';
import { ObjectId } from 'mongodb';
import { Model } from 'mongoose';

import { EVENT, PERFORMER_CHANNEL } from 'src/constants';
import {
  EntityNotFoundException, QueueMessageService, createAlias, isObjectId
} from 'src/core';
import { PerformerDto, ProfileDto } from 'src/dtos';
import { SignedUploadRequestPayload } from 'src/file-module/payloads';
import { FileService } from 'src/file-module/services';
import { PerformerCreatePayload } from 'src/payloads';
import { PerformerUpdatePayload } from 'src/payloads/performer/performer-update.payload';
import { Performer, PerformerDocument } from 'src/schemas';

import { ReactionService } from '../reaction';

@Injectable()
export class PerformerService {
  constructor(
    @InjectModel(Performer.name) private readonly PerformerModel: Model<PerformerDocument>,
    private readonly fileService: FileService,
    @Inject(forwardRef(() => ReactionService))
    private readonly reactionService: ReactionService,
    private readonly queueService: QueueMessageService
  ) { }

  public async generateUsername(name: string, id?: string | ObjectId) {
    // consider if need unique slug with post type
    const username = createAlias(name);
    const query = { username } as any;
    if (id) {
      query._id = { $ne: id };
    }
    const count = await this.PerformerModel.countDocuments(query);
    if (!count) {
      return username;
    }

    return this.generateUsername(`${username}${count}`, id);
  }

  public async create(payload: PerformerCreatePayload, current: ProfileDto): Promise<PerformerDto> {
    const username = await this.generateUsername(payload?.username || payload.name);

    const performer = await this.PerformerModel.create({
      ...payload,
      username,
      createdBy: current._id
    });

    return plainToInstance(PerformerDto, performer.toObject());
  }

  public async findByIdOrUsername(id: string | ObjectId): Promise<PerformerDocument> {
    const query = id instanceof ObjectId || isObjectId(id)
      ? { _id: id }
      : { username: id };
    return this.PerformerModel.findOne(query);
  }

  public async update(id: string | ObjectId, payload: PerformerUpdatePayload): Promise<PerformerDto> {
    const performer = await this.PerformerModel.findById(id);
    if (!performer) throw new EntityNotFoundException();
    const username = await this.generateUsername(payload?.username || payload.name, performer._id);
    merge(performer, { ...payload, username });
    performer.updatedAt = new Date();
    await performer.save();
    return plainToInstance(PerformerDto, performer.toObject());
  }

  public async delete(id: string | ObjectId): Promise<any> {
    const performer = await this.PerformerModel.findById(id);
    if (!performer) throw new EntityNotFoundException();

    await performer.deleteOne();
    const dto = plainToInstance(PerformerDto, performer.toObject());
    await this.queueService.publish(PERFORMER_CHANNEL, {
      eventName: EVENT.DELETED,
      data: dto
    });
    return true;
  }


  public async getDetails(id: string | ObjectId, user?: ProfileDto) {
    const performer = await this.findByIdOrUsername(id);
    if (!performer) throw new EntityNotFoundException();

    const dto = plainToInstance(PerformerDto, performer.toObject());

    if (user && user._id) {
      const { isLiked, isBookmarked, isFollowed } = await this.reactionService.getReactionInfo(dto._id, user);
      dto.isLiked = isLiked;
      dto.isBookmarked = isBookmarked;
      dto.isFollowed = isFollowed;
    }

    if (performer.avatarId) {
      const avatarInfo = await this.getFileUploadedInfo(performer.avatarId.toString());
      if (avatarInfo.url) dto.setAvatarUrl(avatarInfo);
    }

    if (performer.coverId) {
      const coverInfo = await this.getFileUploadedInfo(performer.coverId.toString());
      if (coverInfo.url) dto.setCoverUrl(coverInfo);
    }

    return dto;
  }

  public async getSignedUploadAvatarUrl(payload?: Partial<SignedUploadRequestPayload>) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'performer-avatar',
      mediaType: 'image',
      filename: payload?.filename,
      acl: 'public-read'
    });
    return res;
  }

  public async getSignedUploadCoverUrl(payload?: Partial<SignedUploadRequestPayload>) {
    const res = await this.fileService.getPresignedUploadUrl({
      type: 'performer-cover',
      mediaType: 'image',
      filename: payload?.filename,
      acl: 'public-read'
    });
    return res;
  }

  public async getFileUploadedInfo(fileId: string, authenticated = false, expiresIn = 3600) {
    const file = await this.fileService.getPubicInfo({
      fileId,
      authenticated,
      expiresIn
    });
    if (!file) {
      throw new EntityNotFoundException();
    }
    return file;
  }

  public async createPerformerViaVideoImport(performers: string) {
    const arrayPerformerNames = performers.split(',');
    const ids = [];
    if (arrayPerformerNames.length > 0) {
      await arrayPerformerNames.reduce(async (lp, pName) => {
        await lp;
        const username = createAlias(pName);
        let performer = await this.PerformerModel.findOne({
          username
        });
        if (!performer) {
          performer = await this.PerformerModel.create({
            name: pName,
            username,
            status: 'active'
          });
        }
        ids.push(performer._id);
        return Promise.resolve();
      }, Promise.resolve());
    }
    return ids;
  }

  public async updateStats(performerId: string | ObjectId, stats: Record<string, number>) {
    return this.PerformerModel.updateOne(
      {
        _id: performerId
      },
      {
        $set: stats
      }
    );
  }
}
