import { Inject, Injectable, forwardRef } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model, SortOrder } from 'mongoose';

import { REACTION_ACTION } from 'src/constants';
import { PageableData } from 'src/core';
import { PerformerDto, ProfileDto } from 'src/dtos';
import { FileService } from 'src/file-module/services';
import { PerformerSearchPayload } from 'src/payloads';
import { Performer, PerformerDocument } from 'src/schemas';

import { ReactionService } from '../reaction';
import { VideoService } from '../video';


@Injectable()
export class PerformerSearchService {
  constructor(
    @InjectModel(Performer.name) private readonly PerformerModel: Model<PerformerDocument>,
    @Inject(forwardRef(() => ReactionService))
    private readonly reactionService: ReactionService,
    private readonly fileService: FileService,
    @Inject(forwardRef(() => VideoService))
    private readonly videoService: VideoService
  ) { }

  public async populatePerformerData(data: PerformerDocument[], user?: ProfileDto): Promise<PerformerDto[]> {
    const performerIds = data.map((p) => p._id.toString());
    const avatarIds = data.filter((p) => !!p.avatarId).map((p) => p.avatarId.toString());
    const coverIds = data.filter((p) => !!p.coverId).map((p) => p.coverId.toString());

    const [reactions, files] = await Promise.all([
      user ? this.reactionService.findByQuery({
        objectId: {
          $in: performerIds
        },
        createdBy: user._id
      }) : [],
      this.fileService.getFilesPublicInfo({ fileIds: [...avatarIds, ...coverIds], authenticated: true })
    ]);

    const performers = await Promise.all(data.map(async (p) => {
      const performer = plainToInstance(PerformerDto, p);

      const like = reactions.find((l) => l.objectId.toString() === p._id.toString() && l.action === REACTION_ACTION.LIKE);
      performer.isLiked = !!like;

      if (performer.avatarId && files[performer.avatarId.toString()]) {
        const avatar = files[performer.avatarId.toString()];
        performer.setAvatarUrl(avatar);
      }

      if (performer.coverId && files[performer.coverId.toString()]) {
        const cover = files[performer.coverId.toString()];
        performer.setCoverUrl(cover);
      }

      const totalVideoViews = await this.videoService.countTotalVideoViewsByPerformerId(performer._id);
      performer.stats.totalVideoViews = totalVideoViews;

      return performer;
    }));

    return performers;
  }

  // user for admin search
  public async advancedSearch(payload: PerformerSearchPayload): Promise<PageableData<PerformerDto>> {
    const query: FilterQuery<PerformerDocument> = {};

    if (payload.q) {
      const searchValue = { $search: `"${payload.q.toLowerCase()}"` };
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          $text: searchValue
        },
        {
          name: { $regex: regexp }
        },
        {
          username: payload.q.toLowerCase()
        }
      ];
    }

    if (payload.createdBy) {
      query.createdBy = payload.createdBy;
    }

    if (payload.status) {
      query.status = payload.status;
    }

    const sort: { [key: string]: SortOrder } = {};
    if (payload.sortBy === 'most_videos') {
      sort['stats.videos'] = 'desc';
    }
    sort.updatedAt = 'desc';

    const [data, total] = await Promise.all([
      this.PerformerModel.find(query)
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.PerformerModel.countDocuments(query)
    ]);

    return {
      data: await this.populatePerformerData(data),
      total
    };
  }

  // use for user search
  public async search(payload: PerformerSearchPayload, user?: ProfileDto): Promise<PageableData<PerformerDto>> {
    const query: FilterQuery<PerformerDocument> = {
      status: 'active'
    };

    if (payload.q) {
      query.$or = [
        {
          name: { $regex: payload.q }
        },
        {
          username: { $regex: payload.q }
        }
      ];
    }

    if (payload.createdBy) {
      query.createdBy = payload.createdBy;
    }

    const sort: { [key: string]: SortOrder } = {};
    if (payload.sortBy === 'most_videos') {
      sort['stats.videos'] = 'desc';
    }
    sort.createdAt = 'desc';
    const [data, total] = await Promise.all([
      this.PerformerModel.find(query)
        .sort(sort)
        .limit(payload.limit)
        .skip(payload.offset)
        .lean()
        .exec(),
      this.PerformerModel.countDocuments(query)
    ]);

    return {
      data: await this.populatePerformerData(data, user),
      total
    };
  }

  public async findByIds(performerIds: string[] | ObjectId[], user?: ProfileDto) {
    const data = await this.PerformerModel.find({ _id: { $in: performerIds }, status: 'active' }).lean().exec();
    return this.populatePerformerData(data, user);
  }
}
