import { Injectable, Logger } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { ObjectId } from 'mongodb';
import { FilterQuery, Model } from 'mongoose';

import { REACTION_ACTION } from 'src/constants';
import { PerformerDto, ProfileDto } from 'src/dtos';
import { IGetReactionObjectInfoResponse, IUpdateObjectStats } from 'src/interfaces';
import { ReactionUpdateStatsPayload } from 'src/payloads';
import { Performer, PerformerDocument } from 'src/schemas';

import { PerformerSearchService } from './performer-search.service';
import { AbstractReaction } from '../reaction';

@Injectable()
export class PerformerReactionService extends AbstractReaction {
  private logger = new Logger(PerformerReactionService.name);

  constructor(
    @InjectModel(Performer.name) private readonly PerformerModel: Model<PerformerDocument>,
    private readonly performerSearchService: PerformerSearchService
  ) {
    super();
  }

  public async getReactionObjectInfo(ids: string[], user?: ProfileDto): Promise<IGetReactionObjectInfoResponse<PerformerDto>> {
    const data = await this.PerformerModel.find({ _id: { $in: ids } })
      .lean();
    return {
      items: await this.performerSearchService.populatePerformerData(data, user)
    };
  }

  public async handleUpdateStats(objectId: string | ObjectId, payload: ReactionUpdateStatsPayload): Promise<IUpdateObjectStats> {
    try {
      const performer = await this.PerformerModel.findById(objectId);
      if (!performer) {
        return;
      }
      const { action, value } = payload;
      let fieldToUpdate = '';
      switch (action) {
        case REACTION_ACTION.LIKE:
          fieldToUpdate = 'stats.likes';
          break;
        default:
          break;
      }
      if (!fieldToUpdate) return;

      const updateQuery: FilterQuery<PerformerDocument> = {
        _id: objectId
      };
      if (value < 0) {
        updateQuery[fieldToUpdate] = { $gte: 1 };
      }
      await this.PerformerModel.updateOne(updateQuery, {
        $inc: {
          [fieldToUpdate]: value
        }
      });
    } catch (e) {
      this.logger.error(e);
    }
  }
}
