import { existsSync, readFileSync } from 'fs';
import { join } from 'path';

import { HttpException, Injectable, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { render } from 'mustache';
import { createTransport } from 'nodemailer';

import { SETTING_KEYS } from 'src/constants';
import { QueueService, getFileName } from 'src/core';
import { SendMailPayload } from 'src/payloads';

import { EmailTemplateService } from './email-template.service';
import { SettingService } from '../setting.service';


@Injectable()
export class MailService implements OnModuleInit {
  constructor(
    private readonly configService: ConfigService,
    private readonly settingService: SettingService,
    private readonly queueService: QueueService,
    private readonly emailTemplateService: EmailTemplateService
  ) {
  }

  onModuleInit() {
    this.init();
  }

  private async init() {
    this.queueService.createQueue('mailer_queue');
    this.queueService.processWorker('mailer_queue', this.process.bind(this));
  }

  private async getTransport() {
    const host = SettingService.getValueByKey(SETTING_KEYS.SMTP_HOST);
    const port = SettingService.getValueByKey(SETTING_KEYS.SMTP_PORT);
    const user = SettingService.getValueByKey(SETTING_KEYS.SMTP_USERNAME);
    const pass = SettingService.getValueByKey(SETTING_KEYS.SMTP_PASSWORD);

    if (!host || !port || !user || !pass) {
      throw new HttpException('Invalid confirguration!', 400);
    }
    return createTransport({
      host,
      port,
      secure: port === 465,
      auth: {
        user,
        pass
      },
      tls: {
        rejectUnauthorized: false
      }
    });
  }

  private async getTemplate(template = 'default', isLayout = false): Promise<string> {
    const TEMPLATE_DIR = join(this.configService.get('TEMPLATE_DIR'), 'emails');
    const layout = await this.emailTemplateService.findOne({
      key: isLayout ? `layouts/${template}` : template
    });
    if (layout) return layout.content;

    // eslint-disable-next-line no-param-reassign
    template = getFileName(template, true);

    if (template === 'blank') {
      return isLayout ? '[[BODY]]' : '';
    }

    const layoutFile = isLayout ? join(TEMPLATE_DIR, 'layouts', `${template}.html`) : join(TEMPLATE_DIR, `${template}.html`);
    if (!existsSync(layoutFile)) {
      return isLayout ? '[[BODY]]' : '';
    }

    return readFileSync(layoutFile, 'utf8');
  }

  /**
   * create new profile with unique username, email
   * @param payload
   * @returns
   */
  async process(job: any): Promise<any> {
    const {
      html, template, data, layout, subject, to, cc
    } = job.data as SendMailPayload;

    let emailHtml = html;
    let emailSubject = subject || '';
    if (!emailHtml) {
      emailHtml = await this.getTemplate(template);
    }

    if (template) {
      const emailTemplate = await this.emailTemplateService.findOne({ key: template });
      if (emailTemplate) {
        emailSubject = emailTemplate.subject;
      }
    }

    // send email immediately
    const siteName = await this.settingService.getKeyValue(SETTING_KEYS.SITE_NAME);
    const logoUrl = await this.settingService.getKeyValue(SETTING_KEYS.LOGO_URL);

    const body = render(emailHtml, {
      ...(data || {}),
      siteName,
      logoUrl
    });
    const emailLayout = layout ? this.getTemplate(layout, true) : '[[BODY]]';
    emailHtml = render(emailLayout, {
      subject: emailSubject,
      ...(data || {})
    })
      .replace('[[BODY]]', body);

    const senderConfig = await this.settingService.getKeyValue(SETTING_KEYS.SENDER_EMAIL);
    const senderEmail = senderConfig || this.configService.get('SENDER_EMAIL');

    const transport = await this.getTransport();
    await transport.sendMail({
      from: senderEmail,
      to: Array.isArray(to) ? to.join(',') : to,
      cc: Array.isArray(cc) ? cc.join(',') : cc,
      bcc: Array.isArray(cc) ? cc.join(',') : cc,
      subject: emailSubject,
      html: emailHtml
    });
  }

  public async verify() {
    try {
      const transport = await this.getTransport();
      const siteName = await this.settingService.getKeyValue(SETTING_KEYS.SITE_NAME) || process.env.DOMAIN;
      const senderEmail = await this.settingService.getKeyValue(SETTING_KEYS.SENDER_EMAIL) || process.env.SENDER_EMAIL;
      const adminEmail = await this.settingService.getKeyValue(SETTING_KEYS.ADMIN_EMAIL) || process.env.ADMIN_EMAIL;
      return transport.sendMail({
        from: senderEmail,
        to: adminEmail,
        subject: `Test email ${siteName}`,
        html: 'Hello, this is test email!'
      });
    } catch (e) {
      return {
        hasError: true,
        error: e
      };
    }
  }

  async sendMail(payload: SendMailPayload) {
    this.queueService.add('mailer_queue', 'send_mail', {
      data: payload
    });
  }
}
