import { Injectable } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { plainToInstance } from 'class-transformer';
import { FilterQuery, Model } from 'mongoose';

import { EntityNotFoundException, SearchRequest } from 'src/core';
import { EmailTemplateDto } from 'src/dtos';
import { EmailTemplateUpdatePayload } from 'src/payloads';
import { EmailTemplate, EmailTemplateDocument } from 'src/schemas';

@Injectable()
export class EmailTemplateService {
  constructor(
    @InjectModel(EmailTemplate.name) private readonly EmailTemplateModel: Model<EmailTemplateDocument>
  ) { }

  public async findById(id: string) {
    const template = await this.EmailTemplateModel.findById(id);
    if (!template) throw new EntityNotFoundException();
    return plainToInstance(EmailTemplateDto, template.toObject());
  }

  public async getAllTemplates(payload: SearchRequest) {
    const query: FilterQuery<EmailTemplateDocument> = {};
    if (payload.q) {
      const regexp = new RegExp(
        payload.q.toLowerCase().replace(/[^a-zA-Z0-9\s]/g, ''),
        'i'
      );
      query.$or = [
        {
          name: { $regex: regexp }
        }
      ];
    }
    const templates = await this.EmailTemplateModel.find(query);
    return templates.map((t) => plainToInstance(EmailTemplateDto, t.toObject()));
  }

  public async findOne(query: Record<string, any>) {
    const template = await this.EmailTemplateModel.findOne(query);
    return template ? plainToInstance(EmailTemplateDto, template.toObject()) : null;
  }

  public async updateTemplate(id: string, payload: EmailTemplateUpdatePayload) {
    const template = await this.EmailTemplateModel.findById(id);
    if (!template) throw new EntityNotFoundException();

    template.subject = payload.subject;
    template.content = payload.content;
    template.layout = payload.layout;
    template.updatedAt = new Date();
    template.save();
    return plainToInstance(EmailTemplateDto, template.toObject());
  }
}
